import time
import base64
from typing import Dict, List, Optional, Any
from PyQt6.QtWidgets import (
    QWidget, QVBoxLayout, QPushButton, QLabel, QDialog, 
    QFrame, QHBoxLayout, QLineEdit, QComboBox, QSpacerItem, 
    QSizePolicy, QListWidget, QListWidgetItem, QGridLayout,
    QTabWidget
)
from PyQt6.QtCore import Qt, pyqtSignal, QObject, QThread
from PyQt6.QtGui import QIcon, QFont
from web3 import Web3
from eth_account import Account
from cryptography.fernet import Fernet
import os
import json
import requests
from typing import List, Dict

# zerox_integration.py не используется - используется zerox_v2_integration.py
from config.settings import SettingsManager, settings_manager # Для доступа к ключам API
from plugins.token_price_service import TokenPriceService # Импортируем сервис цен
from plugins.top_tokens_widget import TopTokensWidget # Импортируем виджет топ-токенов
from plugins.modern_wallet_ui import ModernWalletUI # Импортируем современный интерфейс кошелька
# enhanced_wallet_dashboard.py удален - функционал объединен в modern_wallet_ui.py
from plugins.wallet_welcome_ui import WalletWelcomeUI # Импортируем приветственное окно кошелька
from plugins.balance_manager import BalanceManager

# --- Менеджер истории swap'ов ---

class SwapHistoryManager:
    """Управляет сохранением и загрузкой истории swap-транзакций"""
    
    def __init__(self, wallet_address: str):
        self.wallet_address = wallet_address
        self.swaps_file = self._get_swaps_file_path()
        os.makedirs(os.path.dirname(self.swaps_file), exist_ok=True)
    
    def _get_swaps_file_path(self) -> str:
        """Возвращает путь к файлу с историей swap'ов для конкретного кошелька"""
        wallet_short = self.wallet_address[-8:] if self.wallet_address else "default"
        return f"config/wallets/swap_history_{wallet_short}.json"
    
    def save_swap(self, tx_hash: str, from_token: str, from_amount: float, 
                  to_token: str, to_amount: float, network: str, 
                  quote_data: Dict[str, Any]) -> bool:
        """Сохраняет информацию о выполненном swap"""
        try:
            swaps = self.load_all_swaps()
            
            swap_entry = {
                "hash": tx_hash,
                "type": "send",
                "tx_type": "token_swap",
                "from_token": from_token,
                "from_amount": from_amount,
                "to_token": to_token,
                "to_amount": to_amount,
                "network": network,
                "timestamp": int(time.time()),
                "status": "success",
                "from": self.wallet_address,
                "to": quote_data.get('allowanceTarget', quote_data.get('to', 'unknown')),
                "sellToken": quote_data.get('sellToken', ''),
                "buyToken": quote_data.get('buyToken', ''),
                "quote_data": quote_data  # Сохраняем полную котировку для отладки
            }
            
            # Проверяем что такого хеша еще нет
            if not any(s['hash'] == tx_hash for s in swaps):
                swaps.append(swap_entry)
                
                with open(self.swaps_file, 'w', encoding='utf-8') as f:
                    json.dump(swaps, f, ensure_ascii=False, indent=2)
                
                print(f"💾 Swap сохранен: {from_token} → {to_token}, Tx: {tx_hash[:10]}...")
                return True
            else:
                print(f"⚠️ Swap с хешем {tx_hash[:10]}... уже существует")
                return False
                
        except Exception as e:
            print(f"❌ Ошибка сохранения swap: {e}")
            return False
    
    def load_all_swaps(self) -> List[Dict[str, Any]]:
        """Загружает всю историю swap'ов из файла"""
        try:
            if os.path.exists(self.swaps_file):
                with open(self.swaps_file, 'r', encoding='utf-8') as f:
                    return json.load(f)
        except Exception as e:
            print(f"❌ Ошибка загрузки swap истории: {e}")
        
        return []
    
    def get_swap_by_hash(self, tx_hash: str) -> Optional[Dict[str, Any]]:
        """Получает информацию о конкретном swap'е по хешу"""
        swaps = self.load_all_swaps()
        return next((s for s in swaps if s['hash'].lower() == tx_hash.lower()), None)

# --- Классы для работы с кошельком ---

class Web3Wallet:
    def __init__(self, wallet_name=None, password=None):
        self.w3 = Web3()
        self.account = None
        self.wallet_name = wallet_name
        self.wallets_dir = "config/wallets"
        os.makedirs(self.wallets_dir, exist_ok=True)

        if wallet_name and password:
            self.load_wallet(wallet_name, password)

    def create_wallet(self, wallet_name, password):
        self.account = Account.create()
        self.wallet_name = wallet_name
        self.seed_phrase = self._generate_seed_phrase()
        self.save_wallet(wallet_name, password)
        
        # Очищаем seed фразу из памяти после сохранения
        self.seed_phrase = None
        return self.account.address

    def import_wallet(self, private_key, wallet_name, password):
        try:
            self.account = Account.from_key(private_key)
            self.wallet_name = wallet_name
            self.save_wallet(wallet_name, password)
            return self.account.address
        except Exception as e:
            print(f"Ошибка импорта ключа: {e}")
            return None

    def import_from_seed(self, seed_phrase, wallet_name, password):
        """Импорт кошелька из seed фразы"""
        try:
            # Проверяем количество слов в seed фразе
            words = seed_phrase.strip().split()
            if len(words) not in [12, 15, 18, 21, 24]:
                print(f"❌ Неверное количество слов в seed фразе: {len(words)}")
                return None
            
            # Генерируем приватный ключ из seed фразы
            try:
                import mnemonic
                m = mnemonic.Mnemonic("english")
                if not m.check(seed_phrase):
                    print(f"❌ Неверная seed фраза: проверьте правильность ввода")
                    print(f"💡 Убедитесь, что seed фраза содержит валидные BIP39 слова")
                    return None
                
                # Получаем seed из мнемонической фразы (BIP39)
                seed = m.to_seed(seed_phrase)
                
                # Используем правильный метод генерации приватного ключа из seed
                # Создаем HD wallet и получаем первый приватный ключ
                try:
                    from eth_account.hdaccount import HDPath
                    from eth_account.hdaccount.mnemonic import Mnemonic
                    
                    # Создаем HD путь для первого аккаунта (m/44'/60'/0'/0/0)
                    hd_path = HDPath("m/44'/60'/0'/0/0")
                    
                    # Получаем приватный ключ по HD пути
                    private_key = hd_path.derive(seed)
                    
                except ImportError:
                    print(f"⚠️ HD wallet не доступен, используем упрощенный метод")
                    # Упрощенный метод - используем первые 32 байта seed
                    private_key = seed[:32]
                
            except ImportError:
                print(f"⚠️ Библиотека mnemonic не установлена, используем простую генерацию")
                # Простая генерация на основе seed фразы
                import hashlib
                private_key = hashlib.sha256(seed_phrase.encode()).digest()
            
            # Создаем аккаунт из приватного ключа
            self.account = Account.from_key(private_key)
            self.wallet_name = wallet_name
            self.save_wallet(wallet_name, password)
            
            print(f"✅ Кошелек импортирован из seed фразы: {self.account.address}")
            
            # Очищаем чувствительные данные из памяти
            self._clear_sensitive_data()
            return self.account.address
            
        except Exception as e:
            print(f"❌ Ошибка импорта из seed фразы: {e}")
            return None

    def save_wallet(self, wallet_name, password):
        if not self.account:
            return False
        
        # Генерируем ключ шифрования с солью
        key, salt = self._get_key_from_password(password)
        fernet = Fernet(key)
        encrypted_key = fernet.encrypt(self.account.key.hex().encode())
        
        wallet_path = os.path.join(self.wallets_dir, f"{wallet_name}.json")
        with open(wallet_path, 'w') as f:
            json.dump({
                "address": self.account.address,
                "encrypted_key": encrypted_key.decode(),
                "salt": base64.b64encode(salt).decode()  # Сохраняем соль в base64
            }, f)
        
        # Очищаем чувствительные данные из памяти
        self._clear_sensitive_data()
        return True

    def load_wallet(self, wallet_name, password):
        wallet_path = os.path.join(self.wallets_dir, f"{wallet_name}.json")
        if not os.path.exists(wallet_path):
            return False
            
        with open(wallet_path, 'r') as f:
            data = json.load(f)
        
        try:
            # Проверяем есть ли соль в файле (новые кошельки)
            if 'salt' in data:
                # Новый формат с солью
                salt = base64.b64decode(data['salt'].encode())
                key, _ = self._get_key_from_password(password, salt)
            else:
                # Старый формат без соли - используем простой SHA256 для совместимости
                from hashlib import sha256
                import base64
                key = base64.urlsafe_b64encode(sha256(password.encode()).digest())
            
            fernet = Fernet(key)
            decrypted_key = fernet.decrypt(data["encrypted_key"].encode())
            
            # Преобразуем hex строку в байты
            private_key_bytes = bytes.fromhex(decrypted_key.decode())
            self.account = Account.from_key(private_key_bytes)
            self.wallet_name = wallet_name
            
            # Очищаем чувствительные данные из памяти
            self._clear_sensitive_data()
            return True
            
        except Exception as e:
            print(f"❌ Ошибка загрузки кошелька: Неверный пароль или поврежденный файл")
            return False

    def _clear_sensitive_data(self):
        """Очищает чувствительные данные из памяти"""
        # Очищаем seed фразу если она есть
        if hasattr(self, 'seed_phrase'):
            self.seed_phrase = None
        
        # Принудительная сборка мусора для очистки памяти
        import gc
        gc.collect()

    def save_session_state(self, wallet_name, password):
        """Сохраняет состояние сессии кошелька"""
        try:
            session_file = os.path.join(self.wallets_dir, "session.json")
            session_data = {
                "wallet_name": wallet_name,
                "password": password,  # В реальном приложении пароль должен быть зашифрован
                "timestamp": time.time(),
                "is_active": True
            }
            
            with open(session_file, 'w') as f:
                json.dump(session_data, f)
            
            print(f"✅ Состояние сессии сохранено для кошелька: {wallet_name}")
            return True
        except Exception as e:
            print(f"❌ Ошибка сохранения состояния сессии: {e}")
            return False

    def load_session_state(self):
        """Загружает состояние сессии кошелька"""
        try:
            session_file = os.path.join(self.wallets_dir, "session.json")
            
            if not os.path.exists(session_file):
                return None
                
            with open(session_file, 'r') as f:
                session_data = json.load(f)
            
            # Проверяем, что сессия не слишком старая (например, не старше 24 часов)
            current_time = time.time()
            session_age = current_time - session_data.get('timestamp', 0)
            
            if session_age > 24 * 60 * 60:  # 24 часа
                print("⚠️ Сессия устарела, требуется повторный вход")
                self.clear_session_state()
                return None
            
            if session_data.get('is_active', False):
                print(f"✅ Найдена активная сессия для кошелька: {session_data['wallet_name']}")
                return session_data
            
            return None
        except Exception as e:
            print(f"❌ Ошибка загрузки состояния сессии: {e}")
            return None

    def clear_session_state(self):
        """Очищает состояние сессии"""
        try:
            session_file = os.path.join(self.wallets_dir, "session.json")
            if os.path.exists(session_file):
                os.remove(session_file)
                print("✅ Состояние сессии очищено")
        except Exception as e:
            print(f"❌ Ошибка очистки состояния сессии: {e}")

    def auto_login(self):
        """Автоматический вход в кошелек на основе сохраненной сессии"""
        session_data = self.load_session_state()
        
        if not session_data:
            print("🔌 Нет активной сессии для автоматического входа")
            return False
        
        wallet_name = session_data['wallet_name']
        password = session_data['password']
        
        print(f"🔌 Попытка автоматического входа в кошелек: {wallet_name}")
        
        if self.load_wallet(wallet_name, password):
            print(f"✅ Автоматический вход успешен: {self.account.address}")
            return True
        else:
            print("❌ Автоматический вход не удался, очищаем сессию")
            self.clear_session_state()
            return False

    def get_address(self):
        return self.account.address if self.account else None

    @staticmethod
    def get_saved_wallets():
        """Получить список сохраненных кошельков"""
        import os
        import json
        
        wallets_dir = "config/wallets"
        wallets = []
        
        if not os.path.exists(wallets_dir):
            return wallets
            
        try:
            for filename in os.listdir(wallets_dir):
                if filename.endswith('.json'):
                    wallet_name = filename[:-5]  # Убираем .json
                    wallet_path = os.path.join(wallets_dir, filename)
                    
                    # Читаем файл кошелька для получения адреса
                    try:
                        with open(wallet_path, 'r') as f:
                            wallet_data = json.load(f)
                            address = wallet_data.get('address', 'Unknown')
                            wallets.append({
                                'name': wallet_name,
                                'address': address,
                                'path': wallet_path
                            })
                    except Exception as e:
                        print(f"Ошибка чтения кошелька {wallet_name}: {e}")
                        wallets.append({
                            'name': wallet_name,
                            'address': 'Error',
                            'path': wallet_path
                        })
        except Exception as e:
            print(f"Ошибка получения списка кошельков: {e}")
            
        return wallets

    def get_rpc_url(self):
        # Эта логика будет расширена
        return "https://mainnet.infura.io/v3/..."

    def _create_temp_account(self):
        """Создает временный аккаунт для генерации seed фразы"""
        return Account.create()
    
    def _generate_seed_phrase(self):
        """Генерирует мнемоническую фразу из приватного ключа (12 слов)"""
        try:
            import mnemonic
            # Генерируем случайную seed фразу из 12 слов
            m = mnemonic.Mnemonic("english")
            # Генерируем 128 бит энтропии для 12 слов
            import secrets
            entropy = secrets.token_bytes(16)  # 16 байт = 128 бит для 12 слов
            seed_phrase = m.to_mnemonic(entropy)
            # Проверяем количество слов
            word_count = len(seed_phrase.split())
            print(f"🔍 Сгенерирована seed фраза: {word_count} слов")
            return seed_phrase
        except ImportError:
            # Если библиотека mnemonic не установлена, создаем простую фразу
            words = [
                "abandon", "ability", "able", "about", "above", "absent", "absorb", "abstract", "absurd", "abuse",
                "access", "accident", "account", "accuse", "achieve", "acid", "acoustic", "acquire", "across", "act",
                "action", "actor", "actress", "actual", "adapt", "add", "addict", "address", "adjust", "admit",
                "adult", "advance", "advice", "aerobic", "affair", "afford", "afraid", "again", "age", "agent"
            ]
            import random
            seed_phrase = " ".join(random.choices(words, k=12))
            print(f"🔍 Сгенерирована fallback seed фраза: 12 слов")
            return seed_phrase
        except Exception as e:
            print(f"Ошибка генерации seed фразы: {e}")
            fallback_phrase = "abandon ability able about above absent absorb abstract absurd abuse access accident"
            print(f"🔍 Используется fallback seed фраза: 12 слов")
            return fallback_phrase

    @staticmethod
    def _get_key_from_password(password: str, salt: bytes = None) -> tuple:
        """Генерирует криптографически стойкий ключ из пароля с использованием PBKDF2"""
        from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
        from cryptography.hazmat.primitives import hashes
        import base64
        import os
        
        # Если соль не предоставлена, генерируем случайную
        if salt is None:
            salt = os.urandom(16)  # 128 бит соли
        
        # Используем PBKDF2 с SHA256, 100,000 итераций
        kdf = PBKDF2HMAC(
            algorithm=hashes.SHA256(),
            length=32,  # 256 бит ключа
            salt=salt,
            iterations=100000,  # Высокое количество итераций для защиты от брутфорса
        )
        
        key = kdf.derive(password.encode('utf-8'))
        return base64.urlsafe_b64encode(key), salt
    
    def get_current_address(self):
        """Возвращает текущий адрес кошелька"""
        return self.account.address if self.account else None
    
    def get_addresses(self):
        """Возвращает список адресов (для совместимости)"""
        if self.account:
            return [self.account.address]
        return []
    
    def get_balance(self, address: str = None, tokens: list = None, chain: str = "ethereum"):
        """Получает баланс кошелька"""
        try:
            if not address:
                address = self.get_current_address()
            
            if not address:
                return "❌ Кошелек не разблокирован"
            
            # Используем BalanceManager для получения баланса
            from plugins.balance_manager import BalanceManager
            balance_manager = BalanceManager(self, None)
            
            # Переключаем сеть
            network_map = {
                "ethereum": "Ethereum",
                "polygon": "Polygon",
                "bsc": "BSC",
                "arbitrum": "Arbitrum",
                "optimism": "Optimism"
            }
            network_name = network_map.get(chain.lower(), "Polygon")
            balance_manager.set_network(network_name)
            
            # Получаем балансы
            balances_data = balance_manager.get_balances()
            
            if not balances_data:
                return "💰 Баланс: 0"
            
            # Получаем актуальные цены для токенов с балансом
            symbols_to_fetch = []
            if 'native_token' in balances_data and balances_data['native_token']:
                symbols_to_fetch.append(balances_data['native_token']['symbol'])
            if 'tokens' in balances_data:
                for symbol, token_data in balances_data['tokens'].items():
                    if token_data.get('balance', 0) > 0:
                        symbols_to_fetch.append(symbol)
            
            # Получаем цены синхронно
            if symbols_to_fetch:
                try:
                    from plugins.price_fetcher import fetch_prices_sync
                    prices = fetch_prices_sync(symbols_to_fetch)
                    print(f"💵 Получены цены для {len(prices)} токенов: {prices}")
                    
                    # Обновляем USD значения
                    if 'native_token' in balances_data and balances_data['native_token']:
                        symbol = balances_data['native_token']['symbol']
                        if symbol in prices:
                            balance = balances_data['native_token']['balance']
                            balances_data['native_token']['usd_value'] = balance * prices[symbol]
                    
                    if 'tokens' in balances_data:
                        for symbol, token_data in balances_data['tokens'].items():
                            if symbol in prices:
                                balance = token_data['balance']
                                token_data['usd_value'] = balance * prices[symbol]
                except Exception as e:
                    print(f"⚠️ Не удалось получить цены: {e}")
            
            result = f"💰 Баланс кошелька {address[:10]}...{address[-8:]}:\n"
            total_usd = 0.0
            
            # Нативный токен
            if 'native_token' in balances_data and balances_data['native_token']:
                nt = balances_data['native_token']
                usd_val = nt.get('usd_value', 0)
                total_usd += usd_val
                result += f"  {nt['symbol']}: {nt['balance']:.6f} (${usd_val:.2f})\n"
            
            # ERC20 токены  
            if 'tokens' in balances_data:
                for symbol, token_data in balances_data['tokens'].items():
                    if tokens and symbol not in tokens:
                        continue
                    if token_data.get('balance', 0) > 0:  # Показываем только токены с балансом
                        usd_val = token_data.get('usd_value', 0)
                        total_usd += usd_val
                        result += f"  {symbol}: {token_data['balance']:.6f} (${usd_val:.2f})\n"
            
            result += f"\n💵 Общая стоимость: ${total_usd:.2f} USD"
            return result.strip()
            
        except Exception as e:
            print(f"❌ Ошибка получения баланса: {e}")
            import traceback
            traceback.print_exc()
            return f"❌ Ошибка получения баланса: {e}"
    
    def swap_tokens(self, from_token: str, to_token: str, amount: str, slippage_bps: int = 30, chain: str = "ethereum", dry_run: bool = True):
        """Выполняет обмен токенов через 0x API"""
        try:
            if not self.account:
                return "❌ Кошелек не разблокирован. Войдите в кошелек через вкладку 'Кошелек'"
            
            from plugins.zerox_v2_integration import ZeroXV2Integration
            
            # Получаем API ключ 0x
            api_key = None
            if hasattr(self, 'main_window') and hasattr(self.main_window, 'api_manager'):
                api_keys = self.main_window.api_manager.get_api_keys()
                for key_info in api_keys:
                    if key_info.get('provider') == '0x':
                        api_key = key_info.get('key')
                        break
            if not api_key:
                rpc_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'config', 'RPC.txt')
                if os.path.exists(rpc_file):
                    with open(rpc_file, 'r') as f:
                        for line in f:
                            if line.startswith('ZEROX_API_KEY='):
                                api_key = line.strip().split('=', 1)[1]
                                break
            if not api_key:
                api_key = ""
            
            # Конфигурация комиссии из настроек
            wallet_address = self.account.address if self.account else None
            fee_cfg = settings_manager.get_zerox_fee_config(wallet_address) if settings_manager else {}
            fee_recipient = fee_cfg.get("recipient")
            buy_fee = float(fee_cfg.get("buy_token_percentage", 0.0005))
            sell_fee = float(fee_cfg.get("sell_token_percentage", 0.0))
            affiliate_address = fee_cfg.get("affiliate_address") or fee_recipient
            
            # Создаем интеграцию 0x
            zerox_api = ZeroXV2Integration(
                api_key,
                fee_recipient=fee_recipient,
                buy_token_percentage_fee=buy_fee,
                sell_token_percentage_fee=sell_fee,
                affiliate_address=affiliate_address
            )
            
            # Получаем котировку
            print(f"🔄 Получаю котировку для обмена {amount} {from_token} → {to_token}")
            quote = zerox_api.get_quote(
                sell_token=from_token,
                buy_token=to_token,
                sell_amount=amount,
                taker_address=self.account.address,
                slippage_bps=slippage_bps,
                chain_id=chain
            )
            
            if not quote or 'error' in quote:
                error_msg = quote.get('error', 'Неизвестная ошибка') if quote else 'Не удалось получить котировку'
                return f"❌ Ошибка получения котировки: {error_msg}"
            
            # Форматируем результат
            buy_amount = quote.get('buyAmount', '0')
            sell_amount = quote.get('sellAmount', '0')
            
            # Конвертируем из wei с учетом decimals
            from_decimals = quote.get('sellTokenDecimals', 18)
            to_decimals = quote.get('buyTokenDecimals', 18)
            
            from_amount_formatted = float(sell_amount) / (10 ** from_decimals)
            to_amount_formatted = float(buy_amount) / (10 ** to_decimals)
            
            result = f"🔄 Котировка обмена:\n"
            result += f"  Отдаете: {from_amount_formatted:.6f} {from_token}\n"
            result += f"  Получите: {to_amount_formatted:.6f} {to_token}\n"
            result += f"  Курс: 1 {from_token} = {to_amount_formatted/from_amount_formatted:.6f} {to_token}\n"
            
            # Показываем комиссии
            if 'fees' in quote and 'integratorFee' in quote['fees']:
                fee_amount = float(quote['fees']['integratorFee'].get('amount', 0))
                fee_token = quote['fees']['integratorFee'].get('token', '')
                fee_decimals = quote['fees']['integratorFee'].get('decimals', 18)
                fee_formatted = fee_amount / (10 ** fee_decimals)
                result += f"  Комиссия: {fee_formatted:.6f} {fee_token}\n"
            
            if dry_run:
                result += "\n⚠️ Тестовый режим (dry_run=true). Реальный обмен не выполнен."
                result += "\n💡 Для выполнения реального обмена используйте execute=true"
            else:
                result += "\n⚠️ ВНИМАНИЕ: Это реальный обмен! Подтвердите транзакцию."
                result += "\n🚧 Выполнение реального обмена пока в разработке..."
            
            return result
            
        except Exception as e:
            print(f"❌ Ошибка обмена токенов: {e}")
            import traceback
            traceback.print_exc()
            return f"❌ Ошибка обмена токенов: {e}"


# BalanceManager находится в plugins/balance_manager.py
        
class TransactionManager:
    def __init__(self, settings: SettingsManager):
        self.settings = settings
        self.session = requests.Session()
        self.api_configs = {
            "1": {"base_url": "https://api.etherscan.io/api", "api_key_name": "ETHERSCAN_API_KEY"},
            "137": {"base_url": "https://api.polygonscan.com/api", "api_key_name": "POLYGONSCAN_API_KEY"},
        }

    def get_transaction_history(self, address: str, chain_id: str = "1", limit: int = 20) -> List[Dict]:
        if chain_id not in self.api_configs:
            print(f"История транзакций для сети {chain_id} не поддерживается.")
            return []

        config = self.api_configs[chain_id]
        # API ключ можно будет добавить в будущем в настройки
        api_key = "YOUR_API_KEY" 
        
        params = {
            "module": "account",
            "action": "txlist",
            "address": address,
            "startblock": 0,
            "endblock": 99999999,
            "page": 1,
            "offset": limit,
            "sort": "desc",
            "apikey": api_key
        }
        
        try:
            response = self.session.get(config['base_url'], params=params)
            response.raise_for_status()
            data = response.json()
            if data["status"] == "1":
                return data["result"]
            else:
                print(f"Ошибка API {config['base_url']}: {data['message']}")
                return []
        except requests.RequestException as e:
            print(f"Ошибка запроса истории транзакций: {e}")
            return []

# --- Новый дизайн окна обмена ---

class SwapPanel(QWidget):
    def __init__(self, parent=None, wallet=None, main_window=None, current_network=None):
        super().__init__(parent)
        self.wallet = wallet
        self.main_window = main_window
        self.settings_container = main_window.settings_container if main_window else None
        
        # Сохраняем текущую сеть из родительского окна (или используем Polygon по умолчанию)
        self.current_network = current_network if current_network else "Polygon"
        print(f"✅ SwapPanel инициализирован с сетью: {self.current_network}")
        
        # Инициализируем BalanceManager - КРИТИЧЕСКИ ВАЖНО!
        self.balance_manager = BalanceManager(self.wallet, self.settings_container) if self.wallet else None
        if self.balance_manager:
            print(f"✅ BalanceManager инициализирован для SwapPanel")
        else:
            print(f"⚠️ BalanceManager не инициализирован - кошелек не найден")
        
        # Получаем 0x API ключ из настроек
        self.rpc_endpoints = {
            "ethereum": "https://mainnet.infura.io/v3/YOUR_INFURA_ID",
            "polygon": "https://polygon-rpc.com",
        }
        
        self.zerox_api = None
        # Получаем API ключ 0x из config/RPC.txt для новой интеграции v2
        try:
            from plugins.zerox_v2_integration import ZeroXV2Integration
            import os
            
            # Загружаем ключ из config/RPC.txt
            api_key = None
            rpc_file = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'config', 'RPC.txt')
            if os.path.exists(rpc_file):
                with open(rpc_file, 'r') as f:
                    for line in f:
                        if line.startswith('ZEROX_API_KEY='):
                            api_key = line.strip().split('=', 1)[1]
                            break

                    if api_key:
                        wallet_address = None
                        if self.wallet and hasattr(self.wallet, 'account') and self.wallet.account:
                            wallet_address = self.wallet.account.address

                        fee_cfg = settings_manager.get_zerox_fee_config(wallet_address) if settings_manager else {}
                        fee_recipient = fee_cfg.get("recipient")
                        buy_fee = float(fee_cfg.get("buy_token_percentage", 0.0005))
                        sell_fee = float(fee_cfg.get("sell_token_percentage", 0.0))
                        affiliate_address = fee_cfg.get("affiliate_address") or fee_recipient

                        # Используем первую доступную сеть для инициализации
                        rpc_url = "https://polygon.llamarpc.com"  # По умолчанию Polygon
                        self.zerox_api = ZeroXV2Integration(
                            api_key,
                            rpc_url,
                            fee_recipient=fee_recipient,
                            buy_token_percentage_fee=buy_fee,
                            sell_token_percentage_fee=sell_fee,
                            affiliate_address=affiliate_address
                        )
                        print(f"✅ 0x API v2 интеграция инициализирована с ключом из config/RPC.txt")
                    else:
                        print("⚠️ ZEROX_API_KEY не найден в config/RPC.txt")
            else:
                print(f"⚠️ 0x API ключ не найден в config/RPC.txt")
        except Exception as e:
            print(f"❌ Ошибка получения API ключа 0x: {e}")
            import traceback
            traceback.print_exc()

        # Инициализируем менеджер истории swap'ов
        self.swap_history_manager = None
        if wallet and wallet.account:
            self.swap_history_manager = SwapHistoryManager(wallet.account.address)
            print(f"✅ SwapHistoryManager инициализирован для {wallet.account.address[-8:]}")

        self.setup_ui()
        self.connect_signals()
        
    def setup_ui(self):
        self.main_layout = QVBoxLayout(self)
        self.main_layout.setSpacing(5)  # Супер минимальный spacing
        self.main_layout.setContentsMargins(6, 5, 6, 5)  # Супер минимальные отступы

        # Устанавливаем политику размера, чтобы панель растягивалась на всю высоту
        self.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        self.setMinimumHeight(400)  # Ещё меньше минимальная высота

        # 1. Основная панель обмена (токены)
        swap_core = self._create_swap_core_panel()
        self.main_layout.addWidget(swap_core)
        
        # 2. Панель информации о котировке
        quote_panel = self._create_quote_info_panel()
        self.main_layout.addWidget(quote_panel)

        # 3. Кнопки действий
        buttons_panel = self._create_action_buttons_panel()
        self.main_layout.addWidget(buttons_panel)

        # 4. Разделитель
        line = QFrame()
        line.setFrameShape(QFrame.Shape.HLine)
        line.setFrameShadow(QFrame.Shadow.Sunken)
        line.setStyleSheet("background-color: #555555;")
        self.main_layout.addWidget(line)

        # 5. Нижняя панель (История операций) - растягивается на оставшееся место
        bottom_panel = self._create_bottom_panel()
        self.main_layout.addWidget(bottom_panel, 1)  # stretch factor = 1

        # Устанавливаем стили для встроенной панели
        self.setStyleSheet(self._get_stylesheet())

        # НЕ загружаем токены при создании - только при первом открытии формы
        # чтобы не замедлять запуск приложения
        self._tokens_loaded = False

    def _update_token_lists(self):
        """Обновляет списки токенов в комбобоксах в зависимости от текущей сети"""
        if not self.balance_manager:
            print("⚠️ BalanceManager не инициализирован")
            return
        
        # Маппинг сетей UI -> ключи balance_manager
        network_map = {
            'Polygon': 'polygon',
            'Ethereum': 'ethereum',
            'BSC': 'bsc',
            'Arbitrum': 'arbitrum',
            'Optimism': 'optimism',
            'Avalanche': 'avalanche',
            'Base': 'base'
        }
        
        # Получаем ключ текущей сети и chain_id
        current_network_key = network_map.get(self.current_network, 'polygon')
        
        # Получаем информацию о сети
        network_info = self.balance_manager.networks.get(current_network_key)
        if not network_info:
            print(f"⚠️ Информация о сети {self.current_network} не найдена")
            return
        
        chain_id = network_info.get('chain_id')
        
        # Получаем нативный токен для сети
        native_token = network_info.get('native_token', 'ETH')
        
        # Получаем список токенов из zerox_api (более полный список)
        if self.zerox_api and hasattr(self.zerox_api, 'get_available_tokens'):
            # Используем все доступные токены из zerox_v2_integration для текущей сети
            zerox_tokens = self.zerox_api.get_available_tokens(chain_id)
            # Удаляем нативные токены из списка (POL, MATIC, ETH и т.д.)
            zerox_tokens = [t for t in zerox_tokens if t.upper() not in ['POL', 'MATIC', 'ETH', 'BNB', 'AVAX', 'MNT']]
            tokens = [native_token] + zerox_tokens
            print(f"✅ Списки токенов из 0x API для {self.current_network}: {tokens}")
        else:
            # Fallback: используем токены из balance_manager
            token_contracts = self.balance_manager._get_token_contracts_for_network(current_network_key)
            tokens = [native_token] + list(token_contracts.keys())
            print(f"✅ Списки токенов из BalanceManager для {self.current_network}: {tokens}")
        
        # Сохраняем полный список токенов для фильтрации
        self._full_token_list = tokens
        
        # Обновляем комбобоксы
        self.from_token_combo.clear()
        self.from_token_combo.addItems(tokens)
        
        self.to_token_combo.clear()
        self.to_token_combo.addItems(tokens)
        
        print(f"✅ Списки токенов обновлены для сети {self.current_network}: {len(tokens)} токенов: {tokens}")

    # Метод удален - выбор сети теперь в основном окне кошелька

    def _create_swap_core_panel(self):
        panel = QFrame()
        self.swap_core_layout = QVBoxLayout(panel)
        self.swap_core_layout.setSpacing(3)  # Супер минимальный spacing
        self.swap_core_layout.setContentsMargins(0, 0, 0, 0)
        
        self.from_token_panel, self.from_token_combo, self.from_amount_input = self._create_token_panel("Вы отдаете")
        self.to_token_panel, self.to_token_combo, self.to_amount_label = self._create_token_panel("Вы получаете", is_output=True)
        
        # Кнопка смены направления - супер компактная
        self.swap_button = QPushButton("⇅")
        self.swap_button.setFixedSize(24, 24)  # Ещё меньше!
        self.swap_button.setStyleSheet("""
            QPushButton {
                background-color: #4a4a4a;
                color: white;
                border-radius: 12px;
                font-size: 12px;
                font-weight: bold;
                padding: 0px;
            }
            QPushButton:hover {
                background-color: #6c5ce7;
            }
        """)
        
        self.swap_core_layout.addWidget(self.from_token_panel)
        
        # Центральная линия с кнопкой - нулевые отступы
        center_layout = QHBoxLayout()
        center_layout.setContentsMargins(0, 1, 0, 1)  # Почти нулевые отступы
        center_layout.addStretch()
        center_layout.addWidget(self.swap_button)
        center_layout.addStretch()
        self.swap_core_layout.addLayout(center_layout)

        self.swap_core_layout.addWidget(self.to_token_panel)

        return panel

    def _create_token_panel(self, label_text, is_output=False):
        panel = QFrame()
        panel.setObjectName("tokenPanel")
        panel.setMinimumHeight(48)  # Максимально компактно
        panel.setMaximumHeight(52)  # Жёсткое ограничение
        panel.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
        
        layout = QGridLayout(panel)
        layout.setContentsMargins(8, 3, 8, 3)  # Супер минимальные отступы
        layout.setSpacing(3)  # Минимальный spacing
        
        # Метка сверху
        label = QLabel(label_text)
        label.setStyleSheet("font-size: 8px; color: #888888; font-weight: bold;")
        
        token_combo = QComboBox()
        # Включаем редактирование для поиска по тикеру и ввода контракта
        token_combo.setEditable(True)
        token_combo.setInsertPolicy(QComboBox.InsertPolicy.NoInsert)
        # Автодополнение с фильтрацией
        token_combo.setCompleter(None)  # Отключаем стандартный completer
        token_combo.lineEdit().setPlaceholderText("Тикер или 0x...")
        # Токены будут загружены динамически через _update_token_lists()
        token_combo.setMinimumHeight(26)
        token_combo.setMaximumHeight(28)
        token_combo.setMinimumWidth(100)  # Увеличили для адреса
        
        if is_output:
            amount_widget = QLabel("0.0")
            amount_widget.setFont(QFont("Arial", 12, QFont.Weight.Bold))
            amount_widget.setStyleSheet("color: #FFFFFF;")
        else:
            amount_widget = QLineEdit("0.0")
            amount_widget.setFont(QFont("Arial", 12, QFont.Weight.Bold))
            amount_widget.setMinimumHeight(26)
            amount_widget.setMaximumHeight(28)

        layout.addWidget(label, 0, 0, 1, 2)  # Метка на всю ширину
        layout.addWidget(amount_widget, 1, 0)
        layout.addWidget(token_combo, 1, 1)

        return panel, token_combo, amount_widget
        
    def _create_quote_info_panel(self):
        # Создаём единое красивое поле с градиентом
        panel = QFrame()
        panel.setObjectName("quoteInfoPanel")
        panel.setMinimumHeight(75)
        panel.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Minimum)
        panel.setStyleSheet("""
            QFrame#quoteInfoPanel {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #2d2d2d, stop:1 #252525);
                border: 1px solid #3d3d3d;
                border-radius: 8px;
                padding: 8px;
            }
        """)
        
        layout = QVBoxLayout(panel)
        layout.setContentsMargins(10, 8, 10, 8)
        layout.setSpacing(4)
        
        # Строка 1: Котировка
        quote_row = QHBoxLayout()
        quote_row.setSpacing(10)
        kotir_label = QLabel("КОТИРОВКА:")
        kotir_label.setStyleSheet("color: #888888; font-size: 9px; font-weight: bold;")
        self.quote_label = QLabel("N/A")
        self.quote_label.setStyleSheet("color: #00D1B2; font-weight: bold; font-size: 10px;")
        self.quote_label.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignVCenter)
        quote_row.addWidget(kotir_label)
        quote_row.addWidget(self.quote_label, 1)
        
        # Строка 2: Комиссия
        fee_row = QHBoxLayout()
        fee_row.setSpacing(10)
        komis_label = QLabel("КОМИССИЯ:")
        komis_label.setStyleSheet("color: #888888; font-size: 9px; font-weight: bold;")
        self.fee_label = QLabel("N/A")
        self.fee_label.setStyleSheet("color: #FFA500; font-weight: normal; font-size: 9px;")
        self.fee_label.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignVCenter)
        fee_row.addWidget(komis_label)
        fee_row.addWidget(self.fee_label, 1)
        
        # Строка 3: Итого
        total_row = QHBoxLayout()
        total_row.setSpacing(10)
        itogo_label = QLabel("ИТОГО:")
        itogo_label.setStyleSheet("color: #888888; font-size: 9px; font-weight: bold;")
        self.total_label = QLabel("N/A")
        self.total_label.setStyleSheet("color: #6c5ce7; font-weight: bold; font-size: 10px;")
        self.total_label.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignVCenter)
        total_row.addWidget(itogo_label)
        total_row.addWidget(self.total_label, 1)
        
        # Добавляем все строки в вертикальный layout
        layout.addLayout(quote_row)
        layout.addLayout(fee_row)
        layout.addLayout(total_row)
        
        return panel

    def _create_action_buttons_panel(self):
        panel = QFrame()
        layout = QHBoxLayout(panel)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(5)
        
        self.swap_action_button = QPushButton("Обмен")
        self.swap_action_button.setMinimumHeight(16)
        self.swap_action_button.setMaximumHeight(18)
        self.swap_action_button.setStyleSheet("""
            QPushButton {
                background-color: #6c5ce7;
                color: white;
                font-weight: bold;
                font-size: 11px;
                border-radius: 3px;
                padding: 2px 8px;
            }
            QPushButton:hover {
                background-color: #7c6cf7;
            }
        """)
        
        layout.addWidget(self.swap_action_button)
        return panel


    def _create_bottom_panel(self):
        panel = QFrame()
        panel.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        layout = QVBoxLayout(panel)
        layout.setContentsMargins(0, 3, 0, 0)
        layout.setSpacing(3)
        
        # Заголовок
        history_label = QLabel("Последние операции")
        history_label.setStyleSheet("font-size: 10px; font-weight: bold; color: #CCCCCC;")
        layout.addWidget(history_label)
        
        # Список операций - начальная высота поменьше
        self.operations_list = QListWidget()
        self.operations_list.setMinimumHeight(80)  # Уменьшена минимальная высота
        self.operations_list.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        self.operations_list.setStyleSheet("""
            QListWidget {
                background-color: #2c2c2c;
                border: 1px solid #444444;
                border-radius: 4px;
                padding: 3px;
            }
            QListWidget::item {
                padding: 5px;
                border-bottom: 1px solid #3c3c3c;
                font-size: 10px;
            }
            QListWidget::item:hover {
                background-color: #3c3c3c;
            }
        """)
        layout.addWidget(self.operations_list)
        return panel
    
    def update_network(self, network_name):
        """Обновляет текущую сеть (вызывается из родительского окна при переключении)"""
        self.current_network = network_name
        print(f"🔄 SwapPanel: Сеть обновлена на {network_name}")
        # Сбрасываем текущую котировку при смене сети
        self.quote_label.setText("N/A")
        self.fee_label.setText("N/A")
        self.total_label.setText("N/A")
        # Обновляем списки токенов для новой сети
        self._update_token_lists()

    def connect_signals(self):
        self.swap_action_button.clicked.connect(self._handle_swap)
        self.swap_button.clicked.connect(self._swap_token_directions)
        
        # Автоматическое получение котировки при изменении параметров
        self.from_amount_input.textChanged.connect(self._on_amount_changed)
        self.from_token_combo.currentTextChanged.connect(self._on_token_changed)
        self.to_token_combo.currentTextChanged.connect(self._on_token_changed)
    
    def _on_amount_changed(self, text):
        """Вызывается при изменении суммы - автоматически получает котировку"""
        try:
            amount = float(text) if text else 0
            if amount > 0:
                # Небольшая задержка чтобы не спамить API при наборе
                from PyQt6.QtCore import QTimer
                if hasattr(self, '_quote_timer'):
                    self._quote_timer.stop()
                self._quote_timer = QTimer()
                self._quote_timer.setSingleShot(True)
                self._quote_timer.timeout.connect(self._auto_get_quote)
                self._quote_timer.start(800)  # 800ms задержка
        except ValueError:
            pass
    
    def _on_token_changed(self, text):
        """Вызывается при изменении выбора токена - автоматически получает котировку"""
        # Проверяем что это не пустой текст
        if text:
            self._auto_get_quote()
    
    def _auto_get_quote(self):
        """Автоматически получает котировку (без выполнения обмена)"""
        try:
            amount_text = self.from_amount_input.text()
            amount = float(amount_text) if amount_text else 0
            if amount > 0:
                self._get_quote()
        except ValueError:
            pass

    def _handle_swap(self):
        """Выполняет обмен токенов"""
        # Шаг 1: Получаем котировку
        success = self._get_quote()
        
        if not success:
            print("❌ Не удалось получить котировку для обмена")
            return
        
        # Шаг 2: Выполняем обмен
        print("🔄 Выполняем обмен...")
        self._execute_swap()

    def _get_quote(self):
        """Получает котировку для обмена. Возвращает True при успехе, False при ошибке"""
        if not self.zerox_api:
            error_msg = "0x API v2 не настроен"
            self.quote_label.setText(error_msg)
            print(f"❌ {error_msg}")
            return False
        
        if not self.balance_manager:
            error_msg = "BalanceManager не инициализирован"
            self.quote_label.setText(error_msg)
            print(f"❌ {error_msg}")
            return False

        from_token_str = self.from_token_combo.currentText()
        to_token_str = self.to_token_combo.currentText()
        print(f"📊 Запрос котировки: {from_token_str} -> {to_token_str}")
        
        try:
            amount = float(self.from_amount_input.text())
            if amount <= 0: 
                self.quote_label.setText("Введите сумму больше 0")
                return False

            # Получаем текущую сеть из родительского окна кошелька
            # Преобразуем название сети UI в ключ для balance_manager
            network_map = {
                'Polygon': 'polygon',
                'Ethereum': 'ethereum',
                'BSC': 'bsc',
                'Arbitrum': 'arbitrum',
                'Optimism': 'optimism',
                'Avalanche': 'avalanche',
                'Base': 'base'
            }
            current_network_key = network_map.get(self.current_network, 'polygon')
            
            # Получаем информацию о сети
            network_info = self.balance_manager.networks.get(current_network_key)
            if not network_info:
                self.quote_label.setText("Неизвестная сеть")
                return False
            
            chain_id = network_info["chain_id"]
            
            # Обновляем RPC URL для текущей сети
            rpc_url = network_info["rpc_urls"][0]  # Используем первый RPC
            self.zerox_api.update_rpc_url(rpc_url)
            
            # Определяем адреса токенов
            print(f"🔍 Ищем адрес для {from_token_str} в сети {current_network_key}")
            sell_token = self._get_token_address(from_token_str, current_network_key)
            print(f"🔍 Найден адрес для {from_token_str}: {sell_token}")
            
            print(f"🔍 Ищем адрес для {to_token_str} в сети {current_network_key}")
            buy_token = self._get_token_address(to_token_str, current_network_key)
            print(f"🔍 Найден адрес для {to_token_str}: {buy_token}")
            
            if not sell_token:
                error_msg = f"Неизвестный токен: {from_token_str}"
                self.quote_label.setText(error_msg)
                print(f"❌ {error_msg}")
                return False
            
            if not buy_token:
                error_msg = f"Неизвестный токен: {to_token_str}"
                self.quote_label.setText(error_msg)
                print(f"❌ {error_msg}")
                return False
            
            # Получаем decimals для токенов
            # Проверяем если from_token_str - это адрес контракта
            if from_token_str.startswith('0x') and len(from_token_str) == 42:
                token_info = self._get_token_info_from_contract(from_token_str, current_network_key)
                if token_info:
                    sell_decimals = token_info['decimals']
                    from_token_str = token_info['symbol']  # Обновляем на символ токена
                else:
                    sell_decimals = 18  # Fallback
            else:
                sell_decimals = self.zerox_api._get_token_decimals(from_token_str, chain_id)
            
            # Проверяем если to_token_str - это адрес контракта
            if to_token_str.startswith('0x') and len(to_token_str) == 42:
                token_info = self._get_token_info_from_contract(to_token_str, current_network_key)
                if token_info:
                    buy_decimals = token_info['decimals']
                    to_token_str = token_info['symbol']  # Обновляем на символ токена
                else:
                    buy_decimals = 18  # Fallback
            else:
                buy_decimals = self.zerox_api._get_token_decimals(to_token_str, chain_id)
            
            print(f"📊 Decimals: {from_token_str}={sell_decimals}, {to_token_str}={buy_decimals}")
            
            # Получаем адрес пользователя
            taker_address = self.wallet.account.address if self.wallet and self.wallet.account else None
            if not taker_address:
                self.quote_label.setText("Кошелек не подключен")
                return False
            
            print(f"🔄 Получаем котировку: {amount} {from_token_str} -> {to_token_str}")
            print(f"🌐 Сеть: {network_info['name']} (Chain ID: {chain_id})")
            
            # Получаем котировку от 0x API v2
            quote = self.zerox_api.get_quote(
                sell_token=sell_token,
                buy_token=buy_token,
                sell_amount=amount,
                chain_id=chain_id,
                taker=taker_address,
                sell_decimals=sell_decimals
            )

            if not quote:
                self.quote_label.setText("Не удалось получить котировку")
                return False

            # Парсим ответ от API v2
            buy_amount = int(quote.get('buyAmount', 0))
            sell_amount = int(quote.get('sellAmount', 0))
            
            # Обновляем buy_decimals если есть tokenMetadata в ответе
            token_metadata = quote.get('tokenMetadata')
            if token_metadata:
                buy_decimals = self.zerox_api._get_token_decimals(to_token_str, chain_id, token_metadata)
                print(f"📊 Обновлены decimals из tokenMetadata: {to_token_str}={buy_decimals}")
            
            if buy_amount > 0 and sell_amount > 0:
                # Используем правильные decimals для форматирования
                buy_amount_formatted = buy_amount / (10**buy_decimals)
                sell_amount_formatted = sell_amount / (10**sell_decimals)
                
                # Рассчитываем курс: сколько buy_token за 1 sell_token
                rate = buy_amount_formatted / sell_amount_formatted
                
                # Парсим комиссии из ответа API
                fees = quote.get('fees', {})
                fee_details = []
                total_fee_amount = 0
                
                # Gas fee - рассчитываем из transaction.gas и transaction.gasPrice
                transaction = quote.get('transaction', {})
                gas_limit = int(transaction.get('gas', 0))
                gas_price = int(transaction.get('gasPrice', 0))
                if gas_limit > 0 and gas_price > 0:
                    # Gas fee в wei
                    gas_cost_wei = gas_limit * gas_price
                    # Конвертируем в нативный токен
                    gas_formatted = gas_cost_wei / (10**18)
                    fee_details.append(f"Gas: ~{gas_formatted:.4f} {from_token_str}")
                else:
                    # Fallback на gasFee из fees если есть
                    gas_fee = fees.get('gasFee')
                    if gas_fee:
                        gas_amount = int(gas_fee.get('amount', 0))
                        if gas_amount > 0:
                            gas_formatted = gas_amount / (10**18)
                            fee_details.append(f"Gas: ~{gas_formatted:.4f} {from_token_str}")
                
                # 0x Protocol fee
                zerox_fee = fees.get('zeroExFee')
                if zerox_fee:
                    zerox_amount = int(zerox_fee.get('amount', 0))
                    zerox_token = zerox_fee.get('token', '')
                    if zerox_amount > 0:
                        # Конвертируем в правильные единицы
                        zerox_formatted = zerox_amount / (10**buy_decimals)
                        fee_details.append(f"Protocol: {zerox_formatted:.6f} {to_token_str}")
                        total_fee_amount += zerox_formatted
                
                # Integrator fee
                integrator_fee = fees.get('integratorFee')
                if integrator_fee:
                    int_amount = int(integrator_fee.get('amount', 0))
                    int_token = integrator_fee.get('token', '')
                    if int_amount > 0:
                        int_formatted = int_amount / (10**buy_decimals)
                        fee_details.append(f"Fee: {int_formatted:.6f} {to_token_str}")
                        total_fee_amount += int_formatted
                
                # Формируем текст комиссий - полный текст
                if fee_details:
                    fee_text = " | ".join(fee_details)
                else:
                    fee_text = "Комиссия включена"
                
                # minBuyAmount - минимальная сумма с учетом slippage
                min_buy_amount = int(quote.get('minBuyAmount', 0))
                min_buy_formatted = min_buy_amount / (10**buy_decimals)
                
                # Итоговая сумма = buyAmount (уже включает все комиссии в 0x API v2)
                final_amount = buy_amount_formatted
                
                self.quote_label.setText(f"1 {from_token_str} ≈ {rate:.6f} {to_token_str}")
                self.fee_label.setText(fee_text)
                self.total_label.setText(f"~{final_amount:.6f} {to_token_str} (мин: {min_buy_formatted:.4f})")
                self.to_amount_label.setText(f"{final_amount:.6f}")
                
                # Сохраняем данные для выполнения обмена
                self.tx_data = quote
                
                print(f"✅ Котировка получена: {rate:.6f} {to_token_str} за 1 {from_token_str}")
                print(f"💰 Комиссии: {fee_text}")
                print(f"📊 Итого: {final_amount:.6f} {to_token_str}")
                return True
            else:
                self.quote_label.setText("Неверные данные котировки")
                return False
                
        except ValueError:
            self.quote_label.setText("Ошибка ввода суммы")
            return False
        except Exception as e:
            self.quote_label.setText(f"API Ошибка: {str(e)[:50]}")
            print(f"❌ Ошибка получения котировки: {e}")
            return False
    
    def _get_token_address(self, token_symbol: str, network: str) -> Optional[str]:
        """Получает адрес токена для указанной сети"""
        # Проверяем если это адрес контракта (начинается с 0x и длина 42)
        if token_symbol.startswith('0x') and len(token_symbol) == 42:
            print(f"✅ Обнаружен адрес контракта: {token_symbol}")
            return token_symbol
        
        # Нативные токены используют специальный адрес 0xEee...
        if token_symbol.upper() in ['ETH', 'MATIC', 'POL', 'BNB', 'AVAX', 'MNT']:
            return "0xEeeeeEeeeEeEeeEeEeEeeEEEeeeeEeeeeeeeEEeE"
        
        # Сначала пытаемся получить из zerox_api (приоритет)
        if self.zerox_api and hasattr(self.zerox_api, '_get_token_address'):
            # Получаем chain_id для сети
            network_info = self.balance_manager.networks.get(network) if self.balance_manager else None
            if network_info:
                chain_id = network_info.get('chain_id')
                addr = self.zerox_api._get_token_address(token_symbol, chain_id)
                if addr:
                    print(f"✅ Адрес токена {token_symbol} из 0x API: {addr}")
                    return addr
        
        # Fallback: пробуем получить из balance_manager
        if self.balance_manager:
            token_contracts = self.balance_manager._get_token_contracts_for_network(network)
            addr = token_contracts.get(token_symbol.upper())
            if addr:
                print(f"✅ Адрес токена {token_symbol} из BalanceManager: {addr}")
                return addr
        
        print(f"⚠️ Адрес токена {token_symbol} не найден для сети {network}")
        return None
    
    def _get_token_info_from_contract(self, contract_address: str, network: str) -> Optional[Dict]:
        """Получает информацию о токене по адресу контракта через Web3"""
        try:
            if not self.balance_manager:
                return None
            
            network_info = self.balance_manager.networks.get(network)
            if not network_info:
                return None
            
            rpc_url = network_info['rpc_urls'][0]
            from web3 import Web3
            web3 = Web3(Web3.HTTPProvider(rpc_url))
            
            if not web3.is_connected():
                print(f"⚠️ Не удалось подключиться к RPC")
                return None
            
            # ERC20 ABI для получения symbol и decimals
            erc20_abi = [
                {"constant": True, "inputs": [], "name": "symbol", "outputs": [{"name": "", "type": "string"}], "type": "function"},
                {"constant": True, "inputs": [], "name": "decimals", "outputs": [{"name": "", "type": "uint8"}], "type": "function"},
                {"constant": True, "inputs": [], "name": "name", "outputs": [{"name": "", "type": "string"}], "type": "function"}
            ]
            
            contract = web3.eth.contract(address=Web3.to_checksum_address(contract_address), abi=erc20_abi)
            
            symbol = contract.functions.symbol().call()
            decimals = contract.functions.decimals().call()
            name = contract.functions.name().call()
            
            token_info = {
                'symbol': symbol,
                'decimals': decimals,
                'name': name,
                'address': contract_address
            }
            
            print(f"✅ Получена информация о токене: {symbol} ({name}), decimals: {decimals}")
            return token_info
            
        except Exception as e:
            print(f"❌ Ошибка получения информации о токене {contract_address}: {e}")
            return None

    def _swap_token_directions(self):
        from_idx = self.from_token_combo.currentIndex()
        to_idx = self.to_token_combo.currentIndex()
        
        self.from_token_combo.setCurrentIndex(to_idx)
        self.to_token_combo.setCurrentIndex(from_idx)
        
        # Сброс полей
        self.from_amount_input.setText("0.0")
        self.to_amount_label.setText("0.0")
        self.quote_label.setText("N/A")
        self.fee_label.setText("N/A")
        self.total_label.setText("N/A")
        
    def _execute_swap(self):
        if not self.zerox_api or not self.tx_data or not self.wallet:
            print("❌ Недостаточно данных для обмена")
            return
            
        # Получаем приватный ключ из кошелька
        if not self.wallet.account:
            print("❌ Кошелек не подключен")
            return
            
        private_key = self.wallet.account.key.hex()
        
        # Получаем текущую сеть для обновления RPC
        # Преобразуем название сети UI в ключ для balance_manager
        network_map = {
            'Polygon': 'polygon',
            'Ethereum': 'ethereum',
            'BSC': 'bsc',
            'Arbitrum': 'arbitrum',
            'Optimism': 'optimism',
            'Avalanche': 'avalanche',
            'Base': 'base'
        }
        current_network_key = network_map.get(self.current_network, 'polygon')
        current_network_name = self.current_network
        
        if self.balance_manager:
            network_info = self.balance_manager.networks.get(current_network_key)
            if network_info:
                rpc_url = network_info["rpc_urls"][0]
                current_network_name = network_info['name']
                self.zerox_api.update_rpc_url(rpc_url)
                print(f"🌐 Обновлен RPC для сети: {current_network_name}")

        print(f"🔄 Выполняем обмен через 0x API v2...")
        result = self.zerox_api.execute_swap(private_key, self.tx_data)
        
        if result.get("success"):
            tx_hash = result['tx_hash']
            
            # Получаем информацию о токенах для сохранения
            from_token_str = self.from_token_combo.currentText().split()[0]  # Берем первое слово
            to_token_str = self.to_token_combo.currentText().split()[0]
            from_amount_str = self.from_amount_input.text()
            to_amount_str = self.to_amount_label.text()
            
            try:
                from_amount = float(from_amount_str)
                to_amount = float(to_amount_str)
            except:
                from_amount = 0.0
                to_amount = 0.0
            
            # Сохраняем swap в историю
            if self.swap_history_manager:
                self.swap_history_manager.save_swap(
                    tx_hash=tx_hash,
                    from_token=from_token_str,
                    from_amount=from_amount,
                    to_token=to_token_str,
                    to_amount=to_amount,
                    network=current_network_name,
                    quote_data=self.tx_data
                )
                print(f"💾 Swap успешно сохранен в историю")
            
            item_text = f"✅ Обмен выполнен! Tx: {tx_hash[:10]}..."
            self.operations_list.insertItem(0, QListWidgetItem(item_text))
            print(f"✅ Обмен успешно выполнен: {tx_hash}")
        else:
            error_message = result.get("error", "Неизвестная ошибка")
            item_text = f"❌ Ошибка обмена: {error_message}"
            self.operations_list.insertItem(0, QListWidgetItem(item_text))
            print(f"❌ Ошибка обмена: {error_message}")
        
    def _get_stylesheet(self):
        return """
        QDialog {
            background-color: #2E2F30;
            color: #FFFFFF;
        }
        QLabel {
            color: #CCCCCC;
        }
        QLineEdit, QComboBox {
            background-color: #3C3F41;
            color: #FFFFFF;
            border: 1px solid #555555;
            border-radius: 4px;
            padding: 5px;
            font-size: 14px;
        }
        QPushButton {
            background-color: #007ACC;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 4px;
            min-height: 35px;
        }
        QPushButton:hover {
            background-color: #0088DD;
        }
        QPushButton:disabled {
            background-color: #555555;
        }
        QFrame#tokenPanel {
            background-color: #3C3F41;
            border: 1px solid #555555;
            border-radius: 4px;
            min-height: 48px;
            max-height: 52px;
            padding: 3px;
        }
        QFrame#tokenPanel:hover {
            border: 1px solid #6c5ce7;
        }
        QListWidget {
            background-color: #3C3F41;
            border: 1px solid #555555;
            border-radius: 4px;
        }
        """

# --- Старый дашборд кошелька, который вызывает новое окно ---

from plugins.transaction_scanner import TransactionScanner

class WalletDashboard(QWidget):
    def __init__(self, main_window, wallet=None):
        super().__init__()
        self.main_window = main_window
        self.wallet = wallet or self.main_window.wallet
        self.balance_manager = BalanceManager(self.wallet, self.main_window.settings_container) if self.wallet else None
        # Исправляем опечатку: settings_container вместо settings_manager
        self.transaction_manager = TransactionManager(self.main_window.settings_container)
        # Сканер транзакций для корректной истории (native + ERC20)
        self.tx_scanner = TransactionScanner()
        
        # Инициализируем SwapPanel позже
        self.swap_panel = None

        self.setup_ui()
        if self.wallet:
            self._update_wallet_info()
            self._update_transaction_history()
            # Создаем SwapPanel для использования в вкладке Обмена
            self._init_swap_panel()

    def setup_ui(self):
        # Проверяем, есть ли уже layout
        if self.layout() is None:
            self.layout = QVBoxLayout(self)
        else:
            self.layout = self.layout()
        self.layout.setContentsMargins(10, 10, 10, 10)
        
        # Address
        self.address_label = QLabel("Адрес: N/A")
        self.address_label.setWordWrap(True)
        self.layout.addWidget(self.address_label)

        # Balance
        self.balance_label = QLabel("N/A")
        self.layout.addWidget(self.balance_label)

        # Buttons
        buttons_layout = QHBoxLayout()
        self.send_button = QPushButton("Отправить")
        self.receive_button = QPushButton("Получить")
        self.swap_button = QPushButton("Своп")
        buttons_layout.addWidget(self.send_button)
        buttons_layout.addWidget(self.receive_button)
        buttons_layout.addWidget(self.swap_button)
        self.layout.addLayout(buttons_layout)

        # Создаем вкладки
        self.tabs = QTabWidget()
        self.balance_tab = QWidget()
        self.history_tab = QWidget()
        self.swap_tab = QWidget()
        self.send_tab = QWidget()
        self.receive_tab = QWidget()
        
        self.tabs.addTab(self.balance_tab, "Баланс")
        self.tabs.addTab(self.history_tab, "История")
        self.tabs.addTab(self.swap_tab, "Обмен")
        self.tabs.addTab(self.send_tab, "Отправить")
        self.tabs.addTab(self.receive_tab, "Получить")
        
        # Настраиваем вкладку баланса
        self.balance_layout = QVBoxLayout(self.balance_tab)
        self.token_list = QListWidget()
        self.balance_layout.addWidget(self.token_list)

        # Настраиваем вкладку истории
        self.history_layout = QVBoxLayout(self.history_tab)
        self.history_list = QListWidget()
        self.history_layout.addWidget(self.history_list)
        
        # Настраиваем вкладку обмена
        self.swap_layout = QVBoxLayout(self.swap_tab)
        self.swap_panel_container = QWidget()
        self.swap_layout.addWidget(self.swap_panel_container)
        
        # Настраиваем вкладку отправки
        self.send_layout = QVBoxLayout(self.send_tab)
        send_label = QLabel("Отправка токенов")
        self.send_layout.addWidget(send_label)
        self.send_layout.addStretch()
        
        # Настраиваем вкладку получения
        self.receive_layout = QVBoxLayout(self.receive_tab)
        receive_label = QLabel("Получение токенов")
        self.receive_layout.addWidget(receive_label)
        self.receive_layout.addStretch()

        self.layout.addWidget(self.tabs)
        self.layout.addStretch()

        # Connect signals
        self.swap_button.clicked.connect(self._open_swap_tab)
        self.send_button.clicked.connect(self._open_send_tab)
        self.receive_button.clicked.connect(self._open_receive_tab)
        
    def _update_wallet_info(self):
        if not self.wallet:
            self.address_label.setText("Кошелек не загружен")
            return
            
        # ИСПРАВЛЕНИЕ: Используем get_addresses()
        address_list = self.wallet.get_addresses()
        address = address_list[0] if address_list else "N/A"
        self.address_label.setText(f"Адрес: {address}")
        
        # Получаем балансы с правильными параметрами
        network = "Polygon"  # или получить из конфига
        balances_data = self.balance_manager.get_balances_from_all_tokens(address, network)

        if isinstance(balances_data, dict) and "error" in balances_data:
             self.token_list.addItem(f"Ошибка: {balances_data['error']}")
             return

        if not balances_data:
            self.token_list.addItem("Балансы не найдены или равны нулю.")
            return

        total_value = 0
        for token, data in balances_data.items():
            balance_str = f"{data['balance']} {token}"
            value_str = f"(${data['value_usd']})"
            self.token_list.addItem(f"{balance_str: <20} {value_str: >15}")
            total_value += data['value_usd']
            
        self.balance_label.setText(f"Общий баланс: ${total_value:,.2f}")

    def _update_transaction_history(self):
        if not self.wallet:
            return
        
        self.history_list.clear()
        # ИСПРАВЛЕНИЕ: Используем get_addresses()
        address_list = self.wallet.get_addresses()
        if not address_list:
            self.history_list.addItem("Адрес кошелька не найден.")
            return
        address = address_list[0]
        # Определяем сеть для истории
        network_map = {
            'polygon': 'Polygon',
            'ethereum': 'ETH',
            'bsc': 'BSC',
            'arbitrum': 'Arbitrum',
            'base': 'Base',
            'avalanche': 'Avalanche'
        }
        current_network_key = (self.balance_manager.current_network if self.balance_manager else 'polygon')
        network_name = network_map.get(current_network_key, 'Polygon')
        # Получаем историю через универсальный сканер (native + ERC20)
        history = self.tx_scanner.get_transaction_history(address, network_name)
        
        if not history:
            self.history_list.addItem("История транзакций пуста.")
            return
            
        for tx in history:
            try:
                is_send = tx.get('type') == 'send'
                token_symbol = tx.get('token_symbol') or ( 'ETH' if network_name in ['ETH', 'Ethereum'] else 'POL' if network_name == 'Polygon' else '')  # 🔧 ИСПРАВЛЕНИЕ: проверяем и 'ETH' и 'Ethereum'
                value = float(tx.get('value', 0))
                to_addr_full = tx.get('to') or ''
                to_addr_short = (to_addr_full[0:6] + '...' + to_addr_full[-4:]) if to_addr_full else '[создание контракта]'
                from_addr_full = tx.get('from') or ''
                from_addr_short = (from_addr_full[0:6] + '...' + from_addr_full[-4:]) if from_addr_full else ''
                status = tx.get('status', 'success')
                tx_kind = tx.get('tx_type', 'native')  # native or token
                direction = 'Отправлено' if is_send else 'Получено'
                if tx_kind == 'token' and token_symbol:
                    line = f"{direction}: {value:.4f} {token_symbol} → {to_addr_short} ({status})"
                else:
                    native_sym = 'ETH' if network_name in ['ETH', 'Ethereum'] else 'POL' if network_name == 'Polygon' else token_symbol or 'Native'  # 🔧 ИСПРАВЛЕНИЕ: проверяем и 'ETH' и 'Ethereum'
                    line = f"{direction}: {value:.4f} {native_sym} → {to_addr_short} ({status})"
                self.history_list.addItem(line)
            except Exception as e:
                print(f"⚠️ Ошибка форматирования транзакции: {e}, tx: {tx.get('hash', 'unknown')[:10]}...")
                continue

    def _open_swap_tab(self):
        """Открывает вкладку обмена"""
        self.tabs.setCurrentIndex(self.tabs.indexOf(self.swap_tab))

    def _open_send_tab(self):
        """Открывает вкладку отправки"""
        self.tabs.setCurrentIndex(self.tabs.indexOf(self.send_tab))

    def _open_receive_tab(self):
        """Открывает вкладку получения"""
        self.tabs.setCurrentIndex(self.tabs.indexOf(self.receive_tab))

    def _call_parent_swap_widget(self):
        """Открывает встроенную панель обмена токенов"""
        try:
            # Создаем встроенную панель обмена
            self.swap_panel = SwapPanel(self, self.wallet, self.main_window)
            
            # Добавляем панель обмена в интерфейс кошелька
            self._show_swap_panel()
            print("✅ Панель обмена открыта в интерфейсе кошелька")
        except Exception as e:
            print(f"Ошибка открытия панели обмена: {e}")
            # Показываем простое сообщение об ошибке
            from PyQt6.QtWidgets import QMessageBox
            QMessageBox.warning(self, "Ошибка", f"Не удалось открыть панель обмена: {e}")
    
    def _show_swap_panel(self):
        """Показывает панель обмена в интерфейсе кошелька"""
        # Очищаем текущий layout
        self._clear_layout()
        
        # Добавляем панель обмена
        self.layout.addWidget(self.swap_panel)
        
        # Добавляем кнопку "Назад к кошельку"
        back_button = QPushButton("← Назад к кошельку")
        back_button.clicked.connect(self._show_wallet_interface)
        self.layout.addWidget(back_button)
    
    def _clear_layout(self):
        """Безопасно очищает layout от всех виджетов"""
        while self.layout.count():
            item = self.layout.takeAt(0)
            if item and item.widget():
                item.widget().setParent(None)
            elif item and hasattr(item, 'layout'):
                # Если это layout, рекурсивно очищаем его
                self._clear_nested_layout(item.layout())
            # Для QWidgetItem просто удаляем его
            if item:
                del item
    
    def _clear_nested_layout(self, layout):
        """Рекурсивно очищает вложенные layout"""
        if layout:
            while layout.count():
                item = layout.takeAt(0)
                if item and item.widget():
                    item.widget().setParent(None)
                elif item and hasattr(item, 'layout'):
                    self._clear_nested_layout(item.layout())
                # Для QWidgetItem просто удаляем его
                if item:
                    del item
    
    def _show_wallet_interface(self):
        """Возвращает к основному интерфейсу кошелька"""
        # Очищаем layout
        self._clear_layout()
        
        # Восстанавливаем виджеты кошелька
        self._restore_wallet_widgets()
        if self.wallet:
            self._update_wallet_info()
            self._update_transaction_history()
    
    def _restore_wallet_widgets(self):
        """Восстанавливает виджеты кошелька после очистки layout"""
        # Address
        self.address_label = QLabel("Адрес: N/A")
        self.address_label.setWordWrap(True)
        self.layout.addWidget(self.address_label)

        # Balance
        self.balance_label = QLabel("N/A")
        self.layout.addWidget(self.balance_label)

        # Buttons
        buttons_layout = QHBoxLayout()
        self.send_button = QPushButton("Отправить")
        self.receive_button = QPushButton("Получить")
        self.swap_button = QPushButton("Своп")
        buttons_layout.addWidget(self.send_button)
        buttons_layout.addWidget(self.receive_button)
        buttons_layout.addWidget(self.swap_button)
        self.layout.addLayout(buttons_layout)

        # Создаем вкладки
        self.tabs = QTabWidget()
        self.balance_tab = QWidget()
        self.history_tab = QWidget()
        self.swap_tab = QWidget()
        self.send_tab = QWidget()
        self.receive_tab = QWidget()
        
        self.tabs.addTab(self.balance_tab, "Баланс")
        self.tabs.addTab(self.history_tab, "История")
        self.tabs.addTab(self.swap_tab, "Обмен")
        self.tabs.addTab(self.send_tab, "Отправить")
        self.tabs.addTab(self.receive_tab, "Получить")
        
        # Настраиваем вкладку баланса
        self.balance_layout = QVBoxLayout(self.balance_tab)
        self.token_list = QListWidget()
        self.balance_layout.addWidget(self.token_list)

        # Настраиваем вкладку истории
        self.history_layout = QVBoxLayout(self.history_tab)
        self.history_list = QListWidget()
        self.history_layout.addWidget(self.history_list)
        
        # Настраиваем вкладку обмена
        self.swap_layout = QVBoxLayout(self.swap_tab)
        self.swap_panel_container = QWidget()
        self.swap_layout.addWidget(self.swap_panel_container)
        
        # Настраиваем вкладку отправки
        self.send_layout = QVBoxLayout(self.send_tab)
        send_label = QLabel("Отправка токенов")
        self.send_layout.addWidget(send_label)
        self.send_layout.addStretch()
        
        # Настраиваем вкладку получения
        self.receive_layout = QVBoxLayout(self.receive_tab)
        receive_label = QLabel("Получение токенов")
        self.receive_layout.addWidget(receive_label)
        self.receive_layout.addStretch()

        self.layout.addWidget(self.tabs)
        self.layout.addStretch()

        # Connect signals
        self.swap_button.clicked.connect(self._call_parent_swap_widget)

    def force_refresh(self):
        print("🔄 Принудительное обновление WalletDashboard...")
        self.wallet = self.main_window.wallet
        self.balance_manager = BalanceManager(self.wallet, self.main_window.settings_container) if self.wallet else None
        self._update_wallet_info()

    def _init_swap_panel(self):
        """Инициализирует SwapPanel и добавляет его в вкладку Обмена."""
        if self.swap_panel is None:
            self.swap_panel = SwapPanel(self, self.wallet, self.main_window)
            self.swap_layout.addWidget(self.swap_panel)
            print("✅ SwapPanel инициализирован и добавлен в вкладку Обмена")
        else:
            print("⚠️ SwapPanel уже инициализирован.")

# --- Класс плагина ---

class Plugin:
    def __init__(self, main_window):
        self.main_window = main_window
        self.wallet = getattr(main_window, 'wallet', None)
        self.wallet_dashboard = None

    def get_widget(self):
        """Возвращает основной виджет плагина."""
        # Используем тот же метод что и force_recreate_wallet_widget для единообразия
        return self.force_recreate_wallet_widget()

    def force_recreate_wallet_widget(self):
        """Принудительно пересоздает и возвращает виджет кошелька."""
        print("🔌 Плагин: принудительное пересоздание виджета кошелька.")
        self.wallet = self.main_window.wallet
        print(f"🔍 Текущий кошелек: {self.wallet}")
        print(f"🔍 Кошелек имеет account: {hasattr(self.wallet, 'account') if self.wallet else 'False'}")
        print(f"🔍 Account существует: {self.wallet.account is not None if self.wallet and hasattr(self.wallet, 'account') else 'False'}")
        
        # Если кошелек не инициализирован, пробуем автоматический вход
        if not self.wallet or not hasattr(self.wallet, 'account') or not self.wallet.account:
            print("🔌 Кошелек не инициализирован, пробуем автоматический вход...")
            
            # Создаем новый экземпляр Web3Wallet для автоматического входа
            temp_wallet = Web3Wallet()
            if temp_wallet.auto_login():
                print("✅ Автоматический вход успешен!")
                self.wallet = temp_wallet
                self.main_window.wallet = self.wallet
            else:
                print("🔌 Автоматический вход не удался, показываем приветственное окно")
                self.wallet_welcome = WalletWelcomeUI(self.main_window)
                self.wallet_welcome.wallet_created.connect(self.handle_wallet_created)
                self.wallet_welcome.wallet_imported.connect(self.handle_wallet_imported)
                self.wallet_welcome.wallet_unlocked.connect(self.handle_wallet_unlocked)
                return self.wallet_welcome
        
        # Если кошелек инициализирован, показываем основной интерфейс
        if self.wallet and hasattr(self.wallet, 'account') and self.wallet.account:
            print(f"🔌 Кошелек инициализирован: {self.wallet.account.address}")
            print("🔌 Создаем ModernWalletUI")
            self.wallet_dashboard = ModernWalletUI(self.wallet, self.main_window)
            return self.wallet_dashboard
            
        print("🔌 Плагин: кошелек не найден, виджет не создан.")
        return None

    def handle_wallet_created(self, wallet_name, password):
        """Обработка создания нового кошелька"""
        try:
            print(f"🔌 Создание нового кошелька: {wallet_name}")
            self.wallet = Web3Wallet()
            address = self.wallet.create_wallet(wallet_name, password)
            print(f"✅ Кошелек создан успешно: {address}")
            print(f"🔍 Кошелек имеет account: {hasattr(self.wallet, 'account')}")
            print(f"🔍 Account адрес: {self.wallet.account.address if self.wallet.account else 'None'}")
            
            # Обновляем кошелек в главном окне
            self.main_window.wallet = self.wallet
            print(f"🔍 Кошелек установлен в main_window: {self.main_window.wallet is not None}")
            
            # Принудительно пересоздаем виджет кошелька
            print("🔄 Вызываем update_wallet_widget...")
            self.main_window.update_wallet_widget()
            print("✅ update_wallet_widget завершен")
            
        except Exception as e:
            print(f"❌ Ошибка создания кошелька: {e}")
            import traceback
            traceback.print_exc()
            
    def handle_wallet_imported(self, key_or_seed, wallet_name, password):
        """Обработка импорта кошелька"""
        try:
            print(f"🔌 Импорт кошелька: {wallet_name}")
            self.wallet = Web3Wallet()
            
            # Проверяем что передано - seed фраза или приватный ключ
            if len(key_or_seed.split()) > 1:  # Это seed фраза (несколько слов)
                print(f"🔤 Импорт через seed фразу: {len(key_or_seed.split())} слов")
                address = self.wallet.import_from_seed(key_or_seed, wallet_name, password)
            else:  # Это приватный ключ
                print(f"🔑 Импорт через приватный ключ")
                address = self.wallet.import_wallet(key_or_seed, wallet_name, password)
            
            if address:
                print(f"✅ Кошелек импортирован успешно: {address}")
                
                # Обновляем кошелек в главном окне
                self.main_window.wallet = self.wallet
                
                # Принудительно пересоздаем виджет кошелька
                self.main_window.update_wallet_widget()
            else:
                print(f"❌ Не удалось импортировать кошелек")
                
        except Exception as e:
            print(f"❌ Ошибка импорта кошелька: {e}")
            
    def handle_wallet_unlocked(self, wallet_name, password):
        """Обработка входа в кошелек"""
        try:
            print(f"🔌 Вход в кошелек: {wallet_name}")
            self.wallet = Web3Wallet()
            address = self.wallet.load_wallet(wallet_name, password)
            
            if address:
                print(f"✅ Вход в кошелек успешен: {address}")
                
                # Сохраняем состояние сессии для автоматического входа
                self.wallet.save_session_state(wallet_name, password)
                
                # Обновляем кошелек в главном окне
                self.main_window.wallet = self.wallet
                
                # Принудительно пересоздаем виджет кошелька
                self.main_window.update_wallet_widget()
            else:
                print(f"❌ Не удалось войти в кошелек: неверный пароль")
                # Показываем ошибку пользователю
                from PyQt6.QtWidgets import QMessageBox
                msg = QMessageBox()
                msg.setIcon(QMessageBox.Icon.Warning)
                msg.setWindowTitle("Ошибка входа")
                msg.setText("Неверный пароль для кошелька")
                msg.exec()
            
        except Exception as e:
            print(f"❌ Ошибка входа в кошелек: {e}")
            # Показываем ошибку пользователю
            from PyQt6.QtWidgets import QMessageBox
            msg = QMessageBox()
            msg.setIcon(QMessageBox.Icon.Critical)
            msg.setWindowTitle("Ошибка входа")
            msg.setText(f"Ошибка входа в кошелек: {str(e)}")
            msg.exec()

    def execute_ai_command(self, command: str, args: dict) -> dict:
        """
        Выполняет команду, полученную от AI (Deepseek).
        """
        if not self.wallet:
            return {"success": False, "message": "Кошелек не загружен. Пожалуйста, сначала создайте или импортируйте кошелек."}

        if command == "check_balance":
            try:
                balance_manager = BalanceManager(self.wallet, self.main_window.settings_container)
                balances = balance_manager.get_balances()
                if not balances:
                    return {"success": True, "message": "Ваш кошелек пуст."}
                
                response_message = "Ваши балансы:\n"
                for token, data in balances.items():
                    response_message += f"- {data['balance']} {token} (примерно ${data['value_usd']})\n"
                return {"success": True, "message": response_message}
            except Exception as e:
                return {"success": False, "message": f"Ошибка получения баланса: {e}"}

        elif command == "swap_tokens":
            # Эта логика потребует UI-взаимодействия или подтверждения,
            # но пока мы можем инициировать процесс.
            # В будущем здесь можно открывать окно обмена с предзаполненными данными.
            from_token = args.get("from_token")
            to_token = args.get("to_token")
            amount = args.get("amount")
            
            if not all([from_token, to_token, amount]):
                return {"success": False, "message": "Для обмена необходимо указать: from_token, to_token и amount."}
                
            # Просто симулируем начало процесса
            return {"success": True, "message": f"Готовлюсь к обмену {amount} {from_token} на {to_token}. Пожалуйста, подтвердите операцию в окне обмена."}
            
        elif command == "get_transaction_history":
            try:
                tx_manager = TransactionManager(self.main_window.settings_manager)
                history = tx_manager.get_transaction_history(self.wallet.get_address())
                if not history:
                    return {"success": True, "message": "История транзакций пуста."}
                
                response_message = "Последние 5 транзакций:\n"
                for tx in history[:5]:
                    value = Web3.from_wei(int(tx['value']), 'ether')
                    response_message += f"- {'Исходящая' if tx['from'].lower() == self.wallet.get_address().lower() else 'Входящая'} на {value:.4f} ETH\n"
                return {"success": True, "message": response_message}
            except Exception as e:
                return {"success": False, "message": f"Ошибка получения истории: {e}"}

        else:
            return {"success": False, "message": f"Неизвестная команда: {command}"}
