import requests
import json
from PyQt6.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QTableWidget, 
    QTableWidgetItem, QHeaderView, QPushButton, QFrame, QScrollArea
)
from PyQt6.QtCore import Qt, QTimer, pyqtSignal
from PyQt6.QtGui import QFont, QColor, QPalette
from typing import Dict, List, Any
import time

class TopTokensWidget(QWidget):
    """Виджет для отображения топ-50 токенов с курсами"""
    
    # Сигнал для обновления данных
    data_updated = pyqtSignal(object)
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.tokens_data = {}
        self.last_update = 0
        self.update_interval = 300  # 5 минут
        self.setup_ui()
        self.load_tokens_data()
        
        # Таймер для автоматического обновления
        self.update_timer = QTimer()
        self.update_timer.timeout.connect(self.load_tokens_data)
        self.update_timer.start(self.update_interval * 1000)
    
    def setup_ui(self):
        """Настройка интерфейса"""
        layout = QVBoxLayout(self)
        layout.setContentsMargins(10, 10, 10, 10)
        
        # Заголовок
        header_frame = QFrame()
        header_frame.setStyleSheet("""
            QFrame {
                background-color: #2b2b2b;
                border-radius: 8px;
                padding: 10px;
            }
        """)
        
        header_layout = QHBoxLayout(header_frame)
        
        title_label = QLabel("🔥 Топ-50 Криптовалют")
        title_label.setStyleSheet("""
            QLabel {
                color: #ffffff;
                font-size: 18px;
                font-weight: bold;
            }
        """)
        
        self.update_button = QPushButton("🔄 Обновить")
        self.update_button.setStyleSheet("""
            QPushButton {
                background-color: #4CAF50;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 4px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #45a049;
            }
        """)
        self.update_button.clicked.connect(self.load_tokens_data)
        
        header_layout.addWidget(title_label)
        header_layout.addStretch()
        header_layout.addWidget(self.update_button)
        
        layout.addWidget(header_frame)
        
        # Таблица токенов
        self.table = QTableWidget()
        self.table.setColumnCount(6)
        self.table.setHorizontalHeaderLabels([
            "#", "Токен", "Цена", "Изменение 24ч", "Объем 24ч", "Рыночная кап."
        ])
        
        # Настройка стилей таблицы
        self.table.setStyleSheet("""
            QTableWidget {
                background-color: #1e1e1e;
                color: #ffffff;
                border: 1px solid #333333;
                border-radius: 8px;
                gridline-color: #333333;
            }
            QTableWidget::item {
                padding: 8px;
                border-bottom: 1px solid #333333;
            }
            QTableWidget::item:selected {
                background-color: #4CAF50;
            }
            QHeaderView::section {
                background-color: #2b2b2b;
                color: #ffffff;
                padding: 10px;
                border: none;
                font-weight: bold;
            }
        """)
        
        # Настройка заголовков
        header = self.table.horizontalHeader()
        header.setSectionResizeMode(0, QHeaderView.ResizeMode.Fixed)  # #
        header.setSectionResizeMode(1, QHeaderView.ResizeMode.Stretch)  # Токен
        header.setSectionResizeMode(2, QHeaderView.ResizeMode.ResizeToContents)  # Цена
        header.setSectionResizeMode(3, QHeaderView.ResizeMode.ResizeToContents)  # Изменение
        header.setSectionResizeMode(4, QHeaderView.ResizeMode.ResizeToContents)  # Объем
        header.setSectionResizeMode(5, QHeaderView.ResizeMode.ResizeToContents)  # Капитализация
        
        self.table.setColumnWidth(0, 50)
        self.table.setAlternatingRowColors(True)
        self.table.setSelectionBehavior(QTableWidget.SelectionBehavior.SelectRows)
        
        layout.addWidget(self.table)
        
        # Статус обновления
        self.status_label = QLabel("Загрузка данных...")
        self.status_label.setStyleSheet("""
            QLabel {
                color: #888888;
                font-size: 12px;
                padding: 5px;
            }
        """)
        layout.addWidget(self.status_label)
    
    def load_tokens_data(self):
        """Загружает данные о топ-50 токенах"""
        try:
            self.status_label.setText("🔄 Загрузка данных...")
            
            # Используем CoinGecko API для получения данных
            url = "https://api.coingecko.com/api/v3/coins/markets"
            params = {
                'vs_currency': 'usd',
                'order': 'market_cap_desc',
                'per_page': 50,
                'page': 1,
                'sparkline': False,
                'price_change_percentage': '24h'
            }
            
            response = requests.get(url, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            
            if data:
                self.tokens_data = data
                self.update_table()
                self.last_update = time.time()
                self.status_label.setText(f"✅ Обновлено: {time.strftime('%H:%M:%S')}")
                self.data_updated.emit(self.tokens_data)
            else:
                self.status_label.setText("❌ Нет данных")
                
        except Exception as e:
            print(f"Ошибка загрузки данных токенов: {e}")
            self.status_label.setText(f"❌ Ошибка: {str(e)[:50]}...")
    
    def update_table(self):
        """Обновляет таблицу с данными токенов"""
        if not self.tokens_data:
            return
        
        self.table.setRowCount(len(self.tokens_data))
        
        for row, token in enumerate(self.tokens_data):
            # Ранг
            rank_item = QTableWidgetItem(str(token.get('market_cap_rank', row + 1)))
            rank_item.setTextAlignment(Qt.AlignmentFlag.AlignCenter)
            self.table.setItem(row, 0, rank_item)
            
            # Название и символ токена
            name = token.get('name', 'Unknown')
            symbol = token.get('symbol', '').upper()
            token_text = f"{name}\n({symbol})"
            
            token_item = QTableWidgetItem(token_text)
            self.table.setItem(row, 1, token_item)
            
            # Цена
            price = token.get('current_price', 0)
            if price >= 1:
                price_text = f"${price:,.2f}"
            else:
                price_text = f"${price:.6f}"
            
            price_item = QTableWidgetItem(price_text)
            price_item.setTextAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignVCenter)
            self.table.setItem(row, 2, price_item)
            
            # Изменение за 24ч
            change_24h = token.get('price_change_percentage_24h', 0)
            if change_24h is not None:
                if change_24h >= 0:
                    change_text = f"+{change_24h:.2f}%"
                    color = QColor(0, 255, 0)  # Зеленый
                else:
                    change_text = f"{change_24h:.2f}%"
                    color = QColor(255, 0, 0)  # Красный
            else:
                change_text = "N/A"
                color = QColor(128, 128, 128)  # Серый
            
            change_item = QTableWidgetItem(change_text)
            change_item.setTextAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignVCenter)
            change_item.setForeground(color)
            self.table.setItem(row, 3, change_item)
            
            # Объем торгов за 24ч
            volume_24h = token.get('total_volume', 0)
            if volume_24h >= 1_000_000_000:
                volume_text = f"${volume_24h/1_000_000_000:.2f}B"
            elif volume_24h >= 1_000_000:
                volume_text = f"${volume_24h/1_000_000:.2f}M"
            else:
                volume_text = f"${volume_24h:,.0f}"
            
            volume_item = QTableWidgetItem(volume_text)
            volume_item.setTextAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignVCenter)
            self.table.setItem(row, 4, volume_item)
            
            # Рыночная капитализация
            market_cap = token.get('market_cap', 0)
            if market_cap >= 1_000_000_000:
                cap_text = f"${market_cap/1_000_000_000:.2f}B"
            elif market_cap >= 1_000_000:
                cap_text = f"${market_cap/1_000_000:.2f}M"
            else:
                cap_text = f"${market_cap:,.0f}"
            
            cap_item = QTableWidgetItem(cap_text)
            cap_item.setTextAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignVCenter)
            self.table.setItem(row, 5, cap_item)
    
    def get_token_price(self, symbol: str) -> float:
        """Получает цену токена по символу"""
        if not self.tokens_data:
            return 0.0
        
        symbol_upper = symbol.upper()
        for token in self.tokens_data:
            if token.get('symbol', '').upper() == symbol_upper:
                return token.get('current_price', 0.0)
        
        return 0.0
    
    def get_token_data(self, symbol: str) -> Dict[str, Any]:
        """Получает полные данные токена по символу"""
        if not self.tokens_data:
            return {}
        
        symbol_upper = symbol.upper()
        for token in self.tokens_data:
            if token.get('symbol', '').upper() == symbol_upper:
                return token
        
        return {}
