import json
from web3 import Web3
from typing import Dict, Optional

# Минимальный ABI для получения информации о токене ERC20
MINIMAL_ERC20_ABI = json.loads('''
[
    {"constant":true,"inputs":[],"name":"name","outputs":[{"name":"","type":"string"}],"payable":false,"stateMutability":"view","type":"function"},
    {"constant":true,"inputs":[],"name":"symbol","outputs":[{"name":"","type":"string"}],"payable":false,"stateMutability":"view","type":"function"},
    {"constant":true,"inputs":[],"name":"decimals","outputs":[{"name":"","type":"uint8"}],"payable":false,"stateMutability":"view","type":"function"}
]
''')

# Адрес, используемый 0x API для обозначения нативной валюты
ZEROX_NATIVE_ADDRESS = "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee"

# Предустановленные популярные токены (очищенные)
PREDEFINED_TOKENS = {
    1: { # Chain 1
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
        "USDT": {"address": "0xdAC17F958D2ee523a2206206994597C13D831ec7", "decimals": 6, "symbol": "USDT"},
        "USDC": {"address": "0xA0b86991c6218b36c1d19d4a2e9eb0ce3606eb48", "decimals": 6, "symbol": "USDC"},
        "DAI": {"address": "0x6B175474E89094C44Da98b954EedeAC495271d0F", "decimals": 18, "symbol": "DAI"},
        "WETH": {"address": "0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2", "decimals": 18, "symbol": "WETH"},
        "WBTC": {"address": "0x2260FAC5E5542a773Aa44fBCfeDf7C193bc2C599", "decimals": 8, "symbol": "WBTC"},
        "UNI": {"address": "0x1f9840a85d5aF5bf1D1762F925BDADdC4201F984", "decimals": 18, "symbol": "UNI"},
        "LINK": {"address": "0x514910771AF9Ca656af840dff83E8264EcF986CA", "decimals": 18, "symbol": "LINK"},
        "AAVE": {"address": "0x7Fc66500c84A76Ad7e9c93437bFc5Ac33E2DDaE9", "decimals": 18, "symbol": "AAVE"},
        "CRV": {"address": "0xD533a949740bb3306d119CC777fa900bA034cd52", "decimals": 18, "symbol": "CRV"},
        "COMP": {"address": "0xc00e94Cb662C3520282E6f5717214004A7f26888", "decimals": 18, "symbol": "COMP"},
        "MKR": {"address": "0x9f8F72aA9304c8B593d555F12eF6589cC3A579A2", "decimals": 18, "symbol": "MKR"},
        "SNX": {"address": "0xC011a73ee8576Fb46F5E1c5751cA3B9Fe0af2a6F", "decimals": 18, "symbol": "SNX"},
        "YFI": {"address": "0x0bc529c00C6401aEF6D220BE8c6Ea1667F6Ad93e", "decimals": 18, "symbol": "YFI"},
        "1INCH": {"address": "0x111111111117dC0aa78b770fA6A738034120C302", "decimals": 18, "symbol": "1INCH"},
        "SUSHI": {"address": "0x6B3595068778DD592e39A122f4f5a5cF09C90fE2", "decimals": 18, "symbol": "SUSHI"},
        "BAL": {"address": "0xba100000625a3754423978a60c9317c58a424e3D", "decimals": 18, "symbol": "BAL"},
        "LDO": {"address": "0x5A98FcBEA516Cf06857215779Fd812CA3beF1B32", "decimals": 18, "symbol": "LDO"},
        "SHIB": {"address": "0x95aD61b0a150d79219dCF64E1E6Cc01f0B64C4cE", "decimals": 18, "symbol": "SHIB"},
        "PEPE": {"address": "0x6982508145454Ce325dDbE47a25d4ec3d2311933", "decimals": 18, "symbol": "PEPE"},
        "DOGE": {"address": "0x4206931337dc273a630d328dA6441786BfaD668f", "decimals": 8, "symbol": "DOGE"},
        "MATIC": {"address": "0x7D1AfA7B718fb893dB30A3aBc0Cfc608AaCfeBB0", "decimals": 18, "symbol": "MATIC"},
        "AVAX": {"address": "0x85f138bfEE4ef8e540890CFb48F620571d67Eda3", "decimals": 18, "symbol": "AVAX"},
        "FTM": {"address": "0x4E15361FD6b4BB609Fa63C81A2be19d873717870", "decimals": 18, "symbol": "FTM"},
        "BNB": {"address": "0xB8c77482e45F1F44dE1745F52C74426C631bDD52", "decimals": 18, "symbol": "BNB"},
        "ADA": {"address": "0x3EE2200Efb3400fAbB9AacF31297cBdD1d435D47", "decimals": 18, "symbol": "ADA"},
        "DOT": {"address": "0x7083609fCE4d1d8Dc0C979aAe8EdF084B89258db", "decimals": 18, "symbol": "DOT"},
        "SOL": {"address": "0xD31a59c85aE9D8edEFeC411D448f90841571b89c", "decimals": 9, "symbol": "SOL"},
        "ATOM": {"address": "0x8D983cb9388EaC77af0474fA441C4815500Cb7BB", "decimals": 18, "symbol": "ATOM"},
        "NEAR": {"address": "0x85F17Cf997934a597031b2E18a9aB6ebD4B9f6a4", "decimals": 24, "symbol": "NEAR"},
        "ALGO": {"address": "0xAC51066d7bEC65Dc4589368da368b212745d63E8", "decimals": 6, "symbol": "ALGO"},
        "ICP": {"address": "0x8cCe97677b8Dc6F8A3Dd82A3F6C3b8D8c1a7Ec8E", "decimals": 8, "symbol": "ICP"},
        "FIL": {"address": "0x6e1A19F235bE7ED8E3369eF73b196C07257494DE", "decimals": 18, "symbol": "FIL"},
        "VET": {"address": "0x1fdb1764c4C4c4C4C4C4C4C4C4C4C4C4C4C4C4C4", "decimals": 18, "symbol": "VET"},
        "TRX": {"address": "0x50327c6c5a14DCaDE707ABad2E27eB517df87AB5", "decimals": 6, "symbol": "TRX"},
        "XRP": {"address": "0x1d2F0da169ceB9fC7B3144628dB156f3F6c60dBE", "decimals": 18, "symbol": "XRP"},
        "LTC": {"address": "0x4338665CBB7B2485A8855A139b75D5e34AB0DB94", "decimals": 18, "symbol": "LTC"},
        "BCH": {"address": "0x8fF795a6F4D97E7887C79beA79aba5cc76444aDf", "decimals": 18, "symbol": "BCH"},
        "EOS": {"address": "0x56b6fB708fC5732DEC1Afc8D8556423A2EDcCbD6", "decimals": 18, "symbol": "EOS"},
        "BAT": {"address": "0x0D8775F648430679A709E98d2b0Cb6250d2887EF", "decimals": 18, "symbol": "BAT"},
        "ZRX": {"address": "0xE41d2489571d322189246DaFA5ebDe1F4699F498", "decimals": 18, "symbol": "ZRX"},
        "REP": {"address": "0x1985365e9f78359a9B6AD760e32412f4a445E862", "decimals": 18, "symbol": "REP"},
        "KNC": {"address": "0xdd974D5C2e2928deA5F71b9825b8b646686BD200", "decimals": 18, "symbol": "KNC"},
        "LRC": {"address": "0xBBbbCA6A901c926F240b89EacB641d8Aec7AEafD", "decimals": 18, "symbol": "LRC"},
        "OMG": {"address": "0xd26114cd6EE289AccF82350c8d8487fedB8A0C07", "decimals": 18, "symbol": "OMG"},
        "STORJ": {"address": "0xB64ef51C888972c908CFacf59B47C1AfBC0Ab8aC", "decimals": 8, "symbol": "STORJ"},
        "GNT": {"address": "0xa74476443119A942dE498590Fe1f2454d7D4aC0d", "decimals": 18, "symbol": "GNT"},
        "MANA": {"address": "0x0F5D2fB29fb7d3CFeE444a200298f468908cC942", "decimals": 18, "symbol": "MANA"},
        "ENJ": {"address": "0xF629cBd94d3791C9250152BD8dfBDF380E2a3B9c", "decimals": 18, "symbol": "ENJ"},
        "SAND": {"address": "0x3845badAde8e6dDD04FcF6924C4F4F4F4F4F4F4F", "decimals": 18, "symbol": "SAND"},
        "AXS": {"address": "0xBB0E17EF65F82Ab018d8EDd776e8DD940327B28b", "decimals": 18, "symbol": "AXS"},
        "CHZ": {"address": "0x3506424F91fD33084466F402d5D97f05F8e3b4AF", "decimals": 18, "symbol": "CHZ"},
        "FLOW": {"address": "0x5c1474680eb6b70E70B410C4c4F4c4F4c4F4c4F4", "decimals": 18, "symbol": "FLOW"},
        "THETA": {"address": "0x3883f5e181fccaF8410FA61e12b59BAd963fb645", "decimals": 18, "symbol": "THETA"},
        "ICX": {"address": "0xb5A4B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5B5", "decimals": 18, "symbol": "ICX"},
        "DGD": {"address": "0xE0B7927c4aF23765Cb51314A0E0521A9645F0E2A", "decimals": 9, "symbol": "DGD"},
        "FUN": {"address": "0x419D0d8BdD9aF5e606Ae2232ed285Aff190E711b", "decimals": 8, "symbol": "FUN"},
        "MCO": {"address": "0xB63B606Ac810a52cCa15e44bB630fd42D8d1d83d", "decimals": 8, "symbol": "MCO"},
        "REQ": {"address": "0x8f8221aFbB33998d8584A2B05749bA73c37a938a", "decimals": 18, "symbol": "REQ"},
        "RLC": {"address": "0x607F4C5BB672230e8672085532f7e901544a7375", "decimals": 9, "symbol": "RLC"},
        "CVC": {"address": "0x41e5560054824eA6B0732E656E3Ad64E20e94E45", "decimals": 8, "symbol": "CVC"},
        "GNO": {"address": "0x6810e776880C02933D47DB1b9fc05908e5386b96", "decimals": 18, "symbol": "GNO"},
        "ADX": {"address": "0x4470BB87d77b963A013DB939BE332f927f2b992e", "decimals": 4, "symbol": "ADX"},
        "MGO": {"address": "0x40395044Ac3c0C57051906dA938B54BD6557F212", "decimals": 8, "symbol": "MGO"},
        "MTL": {"address": "0xF433089366899D83a9f26A773D59ec7eCF30355e", "decimals": 8, "symbol": "MTL"},
        "EDG": {"address": "0x08711D3B02C8758F2FB3ab4e80228418a7F8e39c", "decimals": 0, "symbol": "EDG"},
        "PAY": {"address": "0xB97048628DB6B661D4C2aA833e95Dbe1A905B280", "decimals": 18, "symbol": "PAY"},
        "ADT": {"address": "0xD0D6D6C5Fe4a677D343cC433536BB717bAe167dD", "decimals": 9, "symbol": "ADT"},
        "BNT": {"address": "0x1F573D6Fb3F13d689FF844B4cE37794d79a7FF1C", "decimals": 18, "symbol": "BNT"},
        "ANT": {"address": "0x960b236A07cf122663c4303350609A66A7B288C0", "decimals": 18, "symbol": "ANT"},
        "DNT": {"address": "0x0AbdAce70D6790235e54B0B6E2D6C002178fB15C", "decimals": 18, "symbol": "DNT"},
        "MLN": {"address": "0xec67005c4E4Ec0de61b8eF4c8e0c4D4D4D4D4D4D", "decimals": 18, "symbol": "MLN"},
        "GUP": {"address": "0xf7B098298f7C69Fc14610cF71D5e02c60792894C", "decimals": 3, "symbol": "GUP"},
        "LUN": {"address": "0xfa05A73FfE78ef8f1a739473e462c54bae6567D9", "decimals": 18, "symbol": "LUN"},
        "RCN": {"address": "0xF970b8E36e23F7fC3FD752EeA86f8Be8D83375A6", "decimals": 18, "symbol": "RCN"},
        "RDN": {"address": "0x255Aa6DF07540Cb5d3d297f0D0D4D4D4D4D4D4D4", "decimals": 18, "symbol": "RDN"},
        "SALT": {"address": "0x4156D3342D5c385a87D264F90653733592000581", "decimals": 8, "symbol": "SALT"},
        "MDA": {"address": "0x51DB5Ad35C671a87207d88fC11d593AC0C8415bd", "decimals": 18, "symbol": "MDA"},
        "BQX": {"address": "0x5Af2Be193a6ABCa9c8817001F51744723aC1ED62", "decimals": 8, "symbol": "BQX"},
        "KIN": {"address": "0x818Fc6C2Ec5986bc6E2CBf00939d90556aB12ce5", "decimals": 18, "symbol": "KIN"},
        "WTC": {"address": "0xb7cB1C96dB6B22b0D3d9536E0108d062BD488F74", "decimals": 18, "symbol": "WTC"},
        "AMB": {"address": "0x4DC3643DbC642b72C158E7F3d2ff232df61cb6Ce", "decimals": 18, "symbol": "AMB"},
    },
    56: { # Chain 56
        "BNB": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "BNB"},
        "USDT": {"address": "0x55d398326f99059fF775485246999027B3197955", "decimals": 18, "symbol": "USDT"},
        "USDC": {"address": "0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d", "decimals": 18, "symbol": "USDC"},
        "BUSD": {"address": "0xe9e7CEA3DedcA5984780Bafc599bD69ADd087D56", "decimals": 18, "symbol": "BUSD"},
        "WBNB": {"address": "0xbb4CdB9CBd36B01bD1cBaEBF2De08d9173bc095c", "decimals": 18, "symbol": "WBNB"},
        "CAKE": {"address": "0x0E09FaBB73Bd3Ade0a17ECC321fD13a19e81cE82", "decimals": 18, "symbol": "CAKE"},
        "ADA": {"address": "0x3EE2200Efb3400fAbB9AacF31297cBdD1d435D47", "decimals": 18, "symbol": "ADA"},
        "DOT": {"address": "0x7083609fCE4d1d8Dc0C979aAe8EdF084B89258db", "decimals": 18, "symbol": "DOT"},
        "LINK": {"address": "0xF8A0BF9cF54Bb92F17374d9e9A321E6a111a51bD", "decimals": 18, "symbol": "LINK"},
        "UNI": {"address": "0xBf5140A22578168FD562DCcF235E5D43A02ce9B1", "decimals": 18, "symbol": "UNI"},
        "AAVE": {"address": "0xfb6115445Bff7b52FeB98650C87f44907E58f802", "decimals": 18, "symbol": "AAVE"},
        "SUSHI": {"address": "0x947950BcC74888a40Ffa1293f1C4c4F4F4F4F4F4", "decimals": 18, "symbol": "SUSHI"},
        "COMP": {"address": "0x52CE071Bd9b1C4B00A0b92D298c512478CaD67e8", "decimals": 18, "symbol": "COMP"},
        "MKR": {"address": "0x5f0Da599BB2ccCfcf6FdfD7D81743B6020864350", "decimals": 18, "symbol": "MKR"},
        "SNX": {"address": "0x9Ac983826058b43079a4eF7f9849f4F4F4F4F4F4", "decimals": 18, "symbol": "SNX"},
        "YFI": {"address": "0x88f1A5ae2A3BF98AEAF342D23B4c108094457F0f", "decimals": 18, "symbol": "YFI"},
        "1INCH": {"address": "0x111111111117dC0aa78b770fA6A738034120C302", "decimals": 18, "symbol": "1INCH"},
        "BAL": {"address": "0xba100000625a3754423978a60c9317c58a424e3D", "decimals": 18, "symbol": "BAL"},
        "LDO": {"address": "0x9864e4c9f042B34be2a674b91d1619A6D23f6f0D", "decimals": 18, "symbol": "LDO"},
        "SHIB": {"address": "0x2859e4544C4bB03966803b044A93563Bd2D0DD4D", "decimals": 18, "symbol": "SHIB"},
        "PEPE": {"address": "0x25d2C80DA64413e1DD9B4cd430f0444f0267E4F2", "decimals": 18, "symbol": "PEPE"},
        "DOGE": {"address": "0xbA2aE424d960c26247Dd6c32edC70B295c744C43", "decimals": 8, "symbol": "DOGE"},
        "MATIC": {"address": "0xCC42724C6683B7E57334c4E856f4c9965ED682bD", "decimals": 18, "symbol": "MATIC"},
        "AVAX": {"address": "0x1CE0c2827e2eF14D5C4f29a68d9a2bD4F4F4F4F4", "decimals": 18, "symbol": "AVAX"},
        "FTM": {"address": "0xAD29AbB318791d579433D831ed122aFeAf29dcfe", "decimals": 18, "symbol": "FTM"},
        "ETH": {"address": "0x2170Ed0880ac9a755fd29b2688956bd959f933f8", "decimals": 18, "symbol": "ETH"},
        "BTCB": {"address": "0x7130d2A12B9BCbFAe4f2634d864A1Ee1Ce3Ead9c", "decimals": 18, "symbol": "BTCB"},
        "SOL": {"address": "0x570A5D26f7765Ecb712C0924E4De545B89fD43dF", "decimals": 18, "symbol": "SOL"},
        "ATOM": {"address": "0x0Eb3a705fc54725037CC9e008bDede697f62F335", "decimals": 18, "symbol": "ATOM"},
        "NEAR": {"address": "0x1Fa4a73a3F0133f0025378af2EfF86964cF4F4F4", "decimals": 18, "symbol": "NEAR"},
        "ALGO": {"address": "0x7426Ab52A0e057691E2544fae9C8222e958b2f79", "decimals": 18, "symbol": "ALGO"},
        "FIL": {"address": "0x0D8Ce2A99Bb6e3B7Da6F4F4F4F4F4F4F4F4F4F4", "decimals": 18, "symbol": "FIL"},
        "VET": {"address": "0x6FDcdfef7c496407cCb0cEC90f9C4aA1B8C4C4C4", "decimals": 18, "symbol": "VET"},
        "TRX": {"address": "0x85EAC5Ac2F758618dFa09bDbe0cf174e7d574D5B", "decimals": 18, "symbol": "TRX"},
        "XRP": {"address": "0x1D2F0da169ceB9fC7B3144628dB156f3F6c60dBE", "decimals": 18, "symbol": "XRP"},
        "LTC": {"address": "0x4338665CBB7B2485A8855A139b75D5e34AB0DB94", "decimals": 18, "symbol": "LTC"},
        "BCH": {"address": "0x8fF795a6F4D97E7887C79beA79aba5cc76444aDf", "decimals": 18, "symbol": "BCH"},
        "EOS": {"address": "0x56b6fB708fC5732DEC1Afc8D8556423A2EDcCbD6", "decimals": 18, "symbol": "EOS"},
        "BAT": {"address": "0x101d82428437127bF4aD4F4F4F4F4F4F4F4F4F4F", "decimals": 18, "symbol": "BAT"},
        "ZRX": {"address": "0x85EAC5Ac2F758618dFa09bDbe0cf174e7d574D5B", "decimals": 18, "symbol": "ZRX"},
        "KNC": {"address": "0xfe56d5892BDffC7BF58f2E84BE1b2C32D21C308b", "decimals": 18, "symbol": "KNC"},
        "MANA": {"address": "0x26433c8122652698e2A53C6ef33aC4D4D4D4D4D4", "decimals": 18, "symbol": "MANA"},
        "ENJ": {"address": "0x7e265387c53a5f3fe0c6e7e2a2b2b2b2b2b2b2b2", "decimals": 18, "symbol": "ENJ"},
        "AXS": {"address": "0x715D400F88C167884bbcc41F5A4F4F4F4F4F4F4F", "decimals": 18, "symbol": "AXS"},
        "CHZ": {"address": "0xF9dCe424D6cB2a2F4F4F4F4F4F4F4F4F4F4F4F4F", "decimals": 18, "symbol": "CHZ"},
    },
    8453: { # Chain 8453
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
        "USDC": {"address": "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913", "decimals": 6, "symbol": "USDC"},
        "DAI": {"address": "0x50c5725949A6F0c72E6C4a641F24049A917DB0Cb", "decimals": 18, "symbol": "DAI"},
    },
    42161: { # Chain 42161
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
        "USDT": {"address": "0xFd086bC7CD5C481DCC9C85ebE478A1C0b69FCbb9", "decimals": 6, "symbol": "USDT"},
        "USDC": {"address": "0xaf88d065e77c8cC2239327C5EDb3A432268e5831", "decimals": 6, "symbol": "USDC"},
        "ARB": {"address": "0x912CE59144191C1204E64559FE8253a0e49E6548", "decimals": 18, "symbol": "ARB"},
    },
    324: { # Chain 324
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
        "USDC": {"address": "0x3355df6D4c9C3035724Fd0e3914dE96A5a83aaf4", "decimals": 6, "symbol": "USDC"},
        "USDT": {"address": "0x493257fD37EDB34451f62EDf8D2a0C418852bA4C", "decimals": 6, "symbol": "USDT"},
    },
    137: { # Chain 137
        "POL": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "POL"},
        "USDT": {"address": "0xc2132D05D31c914a87C6611C10748AEb04B58e8F", "decimals": 6, "symbol": "USDT"},
        "USDC": {"address": "0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174", "decimals": 6, "symbol": "USDC"},
        "DAI": {"address": "0x8f3Cf7ad23Cd3CaDbD9735AFf958023239c6A063", "decimals": 18, "symbol": "DAI"},
        "WPOL": {"address": "0x0d500B1d8E8eF31E21C99d1Db9A6444d3ADf1270", "decimals": 18, "symbol": "WPOL"},
        "WETH": {"address": "0x7ceB23fD6bC0adD59B62aC2d8A7c36f03125CEfD", "decimals": 18, "symbol": "WETH"},
        "LINK": {"address": "0x53E0bca35eC356BD5ddDFebbD1Fc0fD03FaBad39", "decimals": 18, "symbol": "LINK"},
        "UNI": {"address": "0xb33EaAd8d922B1083446DC23f610c2567fB5180f", "decimals": 18, "symbol": "UNI"},
        "AAVE": {"address": "0xD6DF932A45C0f255f85145f286eA0b292B21C90B", "decimals": 18, "symbol": "AAVE"},
        "SUSHI": {"address": "0x0b3F868E0BE5597D5DB7fEB59E1CADBb0fdDa50a", "decimals": 18, "symbol": "SUSHI"},
        "CRV": {"address": "0x172370d5Cd63279eFa6d502DAB29171933a610AF", "decimals": 18, "symbol": "CRV"},
        "BAL": {"address": "0x9a71012B13CA4d3D0Cdc72A177DF3ef03b0E76A3", "decimals": 18, "symbol": "BAL"},
        "COMP": {"address": "0x8505b9d2254A7eE7485368a9Cd4B69F7aD54a420", "decimals": 18, "symbol": "COMP"},
        "MKR": {"address": "0x6f7C932e7684666C9fd1d44527765433e01f61d0", "decimals": 18, "symbol": "MKR"},
        "SNX": {"address": "0x50B728D8D964fd00C2d0AAD81718b71311feF68a", "decimals": 18, "symbol": "SNX"},
        "YFI": {"address": "0xDA537104D6A5edd53c6fBba9A898708E465260b6", "decimals": 18, "symbol": "YFI"},
        "1INCH": {"address": "0x9c2C5fd7b07E95EE044DDeba0E97a665F142394f", "decimals": 18, "symbol": "1INCH"},
        "GHST": {"address": "0x385Eeac5cB8A38A6B5E1b4c8eC4aFd2939e3B8c", "decimals": 18, "symbol": "GHST"},
        "QUICK": {"address": "0x831753DD7087CaC61aB5644b308642cc1c33Dc13", "decimals": 18, "symbol": "QUICK"},
        "FRAX": {"address": "0x45c32fA6DF82ead1e2EF74d17b76547EDdFaFF89", "decimals": 18, "symbol": "FRAX"},
        "WBTC": {"address": "0x1BFD67037B42Cf73acF2047067bd4F2C47D9BfD6", "decimals": 8, "symbol": "WBTC"},
        "SHIB": {"address": "0x6f8F0647510C4Bc0C9C4C4C4C4C4C4C4C4C4C4C4", "decimals": 18, "symbol": "SHIB"},
    },
    43114: { # Chain 43114
        "AVAX": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "AVAX"},
        "USDT": {"address": "0x9702230A8Ea53601f5cD2dc00fDBc13d4dF4A8c7", "decimals": 6, "symbol": "USDT"},
        "USDC": {"address": "0xB97EF9Ef8734C71904D8002F8b6Bc66Dd9c48a6E", "decimals": 6, "symbol": "USDC"},
        "WAVAX": {"address": "0xB31f66AA3C1e785363F0875A1B74E27b85FD66c7", "decimals": 18, "symbol": "WAVAX"},
    },
    10: { # Chain 10
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
        "USDT": {"address": "0x94b008aA00579c1307B0EF2c499aD98a8ce58e58", "decimals": 6, "symbol": "USDT"},
        "USDC": {"address": "0x0b2C639c533813f4Aa9D7837CAf62653d097Ff85", "decimals": 6, "symbol": "USDC"},
        "OP": {"address": "0x4200000000000000000000000000000000000042", "decimals": 18, "symbol": "OP"},
    },
    80094: { # Chain 80094
        "BERA": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "BERA"},
    },
    81457: { # Chain 81457
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
        "USDB": {"address": "0x4300000000000000000000000000000000000003", "decimals": 18, "symbol": "USDB"},
    },
    57073: { # Chain 57073
        "INK": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "INK"},
    },
    59144: { # Chain 59144
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
        "USDC": {"address": "0x176211869cA2b568f2A7D4EE941E073a821EE1ff", "decimals": 6, "symbol": "USDC"},
    },
    5000: { # Chain 5000
        "MNT": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "MNT"},
        "USDT": {"address": "0x201EBa5CC46D216Ce6DC03F6a759e8E766e956aE", "decimals": 6, "symbol": "USDT"},
    },
    34443: { # Chain 34443
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
    },
    10143: { # Chain 10143
        "MON": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "MON"},
    },
    534352: { # Chain 534352
        "ETH": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "ETH"},
        "USDC": {"address": "0x06eFdBFf2a14a7c8E15944D1F4A48F9F95F663A4", "decimals": 6, "symbol": "USDC"},
    },
    130: { # Chain 130
        "UNI": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "UNI"},
    },
    480: { # Chain 480
        "WLD": {"address": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee", "decimals": 18, "symbol": "WLD"},
    },
}

def get_token_info(web3: Web3, token_address: str) -> Optional[Dict]:
    """
    Получает информацию (символ, десятичные знаки) о токене ERC20 по адресу его контракта.
    """
    try:
        contract = web3.eth.contract(address=token_address, abi=MINIMAL_ERC20_ABI)
        symbol = contract.functions.symbol().call()
        decimals = contract.functions.decimals().call()
        return {"symbol": symbol, "decimals": decimals}
    except Exception as e:
        print(f"Ошибка получения информации о токене {token_address}: {e}")
        return None

def is_valid_ethereum_address(address: str) -> bool:
    """
    Проверяет, является ли строка валидным адресом Ethereum.
    """
    if not address or not isinstance(address, str):
        return False
    
    # Убираем префикс 0x если есть
    if address.startswith('0x'):
        address = address[2:]
    
    # Проверяем длину (40 символов для hex)
    if len(address) != 40:
        return False
    
    # Проверяем, что все символы hex
    try:
        int(address, 16)
        return True
    except ValueError:
        return False

def get_web3_for_network(chain_id: int) -> Optional[Web3]:
    """
    Создает Web3 экземпляр для указанной сети.
    """
    rpc_urls = {
        1: "https://eth.llamarpc.com",  # Ethereum
        56: "https://bsc-dataseed.binance.org",  # BSC
        137: "https://polygon-rpc.com",  # Polygon
        8453: "https://mainnet.base.org",  # Base
        42161: "https://arb1.arbitrum.io/rpc",  # Arbitrum
        43114: "https://api.avax.network/ext/bc/C/rpc",  # Avalanche
        10: "https://mainnet.optimism.io",  # Optimism
    }
    
    rpc_url = rpc_urls.get(chain_id)
    if not rpc_url:
        print(f"❌ Неизвестная сеть: {chain_id}")
        return None
    
    try:
        web3 = Web3(Web3.HTTPProvider(rpc_url))
        if web3.is_connected():
            return web3
        else:
            print(f"❌ Не удалось подключиться к {rpc_url}")
            return None
    except Exception as e:
        print(f"❌ Ошибка создания Web3 для сети {chain_id}: {e}")
        return None
