#!/usr/bin/env python3
"""
Модуль защиты устройств и сети для AIagent
Интеллектуальная блокировка вредоносных угроз без блокировки системных процессов
"""

import os
import sys
import psutil
import platform
import subprocess
import json
import time
import hashlib
import threading
import logging
from typing import Dict, List, Any, Optional, Set, Tuple
from datetime import datetime, timedelta
from dataclasses import dataclass
from enum import Enum
import re
import socket
import shutil
from pathlib import Path

class ThreatLevel(Enum):
    """Уровни угроз"""
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    CRITICAL = "critical"

class ProtectionType(Enum):
    """Типы защиты"""
    NETWORK = "network"
    PROCESS = "process"
    FILE = "file"
    DEVICE = "device"
    MEMORY = "memory"

class ThreatType(Enum):
    """Типы угроз"""
    MALWARE = "malware"
    SUSPICIOUS_ACTIVITY = "suspicious_activity"
    NETWORK_ATTACK = "network_attack"
    UNAUTHORIZED_ACCESS = "unauthorized_access"
    DATA_EXFILTRATION = "data_exfiltration"

@dataclass
class SecurityThreat:
    """Информация об угрозе безопасности"""
    id: str
    type: ThreatType
    level: ThreatLevel
    source: str
    description: str
    timestamp: datetime
    blocked: bool
    details: Dict[str, Any]

@dataclass
class ProtectionResult:
    """Результат операции защиты"""
    success: bool
    threats_detected: int
    threats_blocked: int
    warnings: List[str]
    details: Dict[str, Any]
    timestamp: datetime

class SecurityProtection:
    """
    Модуль интеллектуальной защиты устройств и сети
    Автоматически блокирует угрозы без нарушения работы системы
    """
    
    def __init__(self):
        self.logger = self._setup_logging()
        self.protection_active = False
        self.monitoring_thread = None
        self.threats_history = []
        self.blocked_ips = set()
        self.blocked_processes = set()
        self.suspicious_files = set()
        self.whitelist = self._load_whitelist()
        self.blacklist = self._load_blacklist()
        self.protection_config = self._load_protection_config()
        
        # Системные процессы (не блокировать)
        self.system_processes = self._get_system_processes()
        
        # Пользовательские процессы (не блокировать)
        self.user_processes = self._get_user_processes()
        
    def _setup_logging(self) -> logging.Logger:
        """Настраивает логирование для модуля защиты"""
        logger = logging.getLogger('SecurityProtection')
        logger.setLevel(logging.INFO)
        
        if not logger.handlers:
            handler = logging.StreamHandler()
            formatter = logging.Formatter(
                '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
            )
            handler.setFormatter(formatter)
            logger.addHandler(handler)
        
        return logger
    
    def _load_whitelist(self) -> Dict[str, Set[str]]:
        """Загружает белый список разрешенных элементов"""
        whitelist = {
            'processes': set(),
            'ips': set(),
            'files': set(),
            'domains': set()
        }
        
        # Добавляем доверенные IP адреса для кошелька и Web3
        whitelist['ips'].update([
            '127.0.0.1', 'localhost', '::1',  # Локальные адреса
            '8.8.8.8', '8.8.4.4',  # Google DNS
            '1.1.1.1', '1.0.0.1',  # Cloudflare DNS
            '208.67.222.222', '208.67.220.220'  # OpenDNS
        ])
        
        # Добавляем доверенные домены для кошелька и Web3
        whitelist['domains'].update([
            'google.com', 'github.com', 'etherscan.io', 'polygonscan.com',
            'bscscan.com', 'basescan.org', 'arbiscan.io', 'snowtrace.io',
            'infura.io', 'ankr.com', 'alchemy.com', 'quicknode.com'
        ])
        
        # Добавляем системные процессы в белый список
        whitelist['processes'].update([
            'python', 'python3', 'bash', 'sh', 'systemd', 'init',
            'kthreadd', 'ksoftirqd', 'migration', 'watchdog'
        ])
        
        # Добавляем процессы кошелька и Web3 в белый список
        whitelist['processes'].update([
            'main', 'aiagent', 'wallet', 'web3', 'metamask', 'trust',
            'chrome', 'firefox', 'brave', 'edge', 'safari',
            'qt', 'pyqt', 'tkinter', 'gtk', 'kde', 'gnome',
            'cursor', 'vscode', 'sublime', 'atom', 'vim', 'nano'
        ])
        
        # Добавляем все процессы Python в белый список
        whitelist['processes'].update([
            'python', 'python3', 'pip', 'pip3', 'main.py', 'aiagent.py'
        ])
        
        return whitelist
    
    def _load_blacklist(self) -> Dict[str, Set[str]]:
        """Загружает черный список заблокированных элементов"""
        blacklist = {
            'processes': set(),
            'ips': set(),
            'files': set(),
            'domains': set()
        }
        
        # Добавляем известные вредоносные паттерны
        blacklist['processes'].update([
            'keylogger', 'backdoor', 'trojan', 'virus'
        ])
        
        return blacklist
    
    def _load_protection_config(self) -> Dict[str, Any]:
        """Загружает конфигурацию защиты"""
        return {
            'enable_network_protection': True,
            'enable_process_protection': True,
            'enable_file_protection': True,
            'enable_device_protection': True,
            'auto_block_threats': True,
            'threat_detection_sensitivity': 'medium',
            'max_blocked_connections': 1000,
            'max_blocked_processes': 100,
            'scan_interval_seconds': 30,
            'enable_heuristic_detection': True,
            'enable_signature_detection': True,
            'enable_behavior_analysis': True
        }
    
    def _get_system_processes(self) -> Set[str]:
        """Получает список системных процессов"""
        system_processes = set()
        
        try:
            # Системные процессы Linux
            if platform.system() == 'Linux':
                system_dirs = ['/bin', '/sbin', '/usr/bin', '/usr/sbin', '/usr/local/bin']
                for sys_dir in system_dirs:
                    if os.path.exists(sys_dir):
                        for item in os.listdir(sys_dir):
                            if os.path.isfile(os.path.join(sys_dir, item)):
                                system_processes.add(item)
            
            # Системные процессы Windows
            elif platform.system() == 'Windows':
                system_processes.update([
                    'svchost.exe', 'explorer.exe', 'winlogon.exe', 'services.exe',
                    'lsass.exe', 'wininit.exe', 'csrss.exe', 'smss.exe'
                ])
            
            # Добавляем процессы Python
            system_processes.update(['python', 'python3', 'pip', 'pip3'])
            
            # Добавляем процессы кошелька и Web3 в системные процессы
            system_processes.update(['main', 'aiagent', 'wallet', 'web3', 'metamask', 'trust'])
            
        except Exception as e:
            self.logger.error(f"Ошибка получения системных процессов: {e}")
        
        return system_processes
    
    def _get_user_processes(self) -> Set[str]:
        """Получает список пользовательских процессов"""
        user_processes = set()
        
        try:
            # Получаем текущие пользовательские процессы
            for proc in psutil.process_iter(['pid', 'name', 'username']):
                try:
                    # Получаем информацию о процессе
                    proc_info = proc.as_dict(attrs=['pid', 'name', 'username'])
                    if proc_info.get('username') == os.getenv('USER'):
                        user_processes.add(proc_info.get('name', 'Unknown'))
                except (psutil.NoSuchProcess, psutil.AccessDenied, psutil.ZombieProcess):
                    continue
        except Exception as e:
            self.logger.error(f"Ошибка получения пользовательских процессов: {e}")
        
        # Добавляем процессы кошелька и Web3 в пользовательские процессы
        user_processes.update([
            'main', 'aiagent', 'wallet', 'web3', 'metamask', 'trust',
            'chrome', 'firefox', 'brave', 'edge', 'safari', 'cursor', 'vscode'
        ])
        
        return user_processes
    
    def start_protection(self) -> ProtectionResult:
        """Запускает защиту"""
        try:
            if self.protection_active:
                return ProtectionResult(
                    success=False,
                    threats_detected=0,
                    threats_blocked=0,
                    warnings=["Защита уже активна"],
                    details={},
                    timestamp=datetime.now()
                )
            
            self.logger.info("Запуск модуля защиты безопасности")
            
            # Запускаем основной цикл мониторинга
            self.protection_active = True
            self.monitoring_thread = threading.Thread(target=self._monitoring_loop, daemon=True)
            self.monitoring_thread.start()
            
            self.logger.info("Защита безопасности активирована")
            
            return ProtectionResult(
                success=True,
                threats_detected=0,
                threats_blocked=0,
                warnings=[],
                details={'status': 'active'},
                timestamp=datetime.now()
            )
            
        except Exception as e:
            self.logger.error(f"Ошибка запуска защиты: {e}")
            return ProtectionResult(
                success=False,
                threats_detected=0,
                threats_blocked=0,
                warnings=[f"Ошибка запуска: {e}"],
                details={},
                timestamp=datetime.now()
            )
    
    def stop_protection(self) -> ProtectionResult:
        """Останавливает защиту"""
        try:
            if not self.protection_active:
                return ProtectionResult(
                    success=False,
                    threats_detected=0,
                    threats_blocked=0,
                    warnings=["Защита не активна"],
                    details={},
                    timestamp=datetime.now()
                )
            
            self.logger.info("Остановка модуля защиты безопасности")
            
            # Останавливаем мониторинг
            self.protection_active = False
            
            if self.monitoring_thread and self.monitoring_thread.is_alive():
                self.monitoring_thread.join(timeout=5)
            
            self.logger.info("Защита безопасности остановлена")
            
            return ProtectionResult(
                success=True,
                threats_detected=0,
                threats_blocked=0,
                warnings=[],
                details={'status': 'stopped'},
                timestamp=datetime.now()
            )
            
        except Exception as e:
            self.logger.error(f"Ошибка остановки защиты: {e}")
            return ProtectionResult(
                success=False,
                threats_detected=0,
                threats_blocked=0,
                warnings=[f"Ошибка остановки: {e}"],
                details={},
                timestamp=datetime.now()
            )
    
    def _monitoring_loop(self):
        """Основной цикл мониторинга безопасности"""
        while self.protection_active:
            try:
                # Сканируем сеть
                if self.protection_config['enable_network_protection']:
                    self._scan_network_threats()
                
                # Сканируем процессы
                if self.protection_config['enable_process_protection']:
                    self._scan_process_threats()
                
                # Сканируем файлы
                if self.protection_config['enable_file_protection']:
                    self._scan_file_threats()
                
                # Сканируем устройства
                if self.protection_config['enable_device_protection']:
                    self._scan_device_threats()
                
                # Ждем следующего сканирования
                time.sleep(self.protection_config['scan_interval_seconds'])
                
            except Exception as e:
                self.logger.error(f"Ошибка в цикле мониторинга: {e}")
                time.sleep(10)  # Пауза при ошибке
    
    def _scan_network_threats(self):
        """Сканирует сетевые угрозы"""
        try:
            # Получаем активные сетевые соединения
            connections = psutil.net_connections()
            
            for conn in connections:
                if not conn.raddr:  # Пропускаем локальные соединения
                    continue
                
                remote_ip = conn.raddr.ip
                remote_port = conn.raddr.port
                
                # Пропускаем доверенные IP адреса
                if remote_ip in self.whitelist['ips']:
                    continue
                
                # Пропускаем локальные адреса
                if remote_ip in ['127.0.0.1', 'localhost', '::1']:
                    continue
                
                # Пропускаем соединения процессов кошелька и Web3
                try:
                    if conn.pid:
                        proc = psutil.Process(conn.pid)
                        proc_name = proc.name() if hasattr(proc, 'name') else 'Unknown'
                        
                        if proc_name in ['main', 'aiagent', 'wallet', 'web3', 'metamask', 'trust']:
                            continue  # Пропускаем соединения кошелька
                        
                        if proc_name in ['chrome', 'firefox', 'brave', 'edge', 'safari', 'cursor', 'vscode']:
                            continue  # Пропускаем соединения браузеров и IDE
                except:
                    pass
                
                # Проверяем IP в черном списке
                if remote_ip in self.blacklist['ips']:
                    self._block_network_connection(conn, f"IP в черном списке: {remote_ip}")
                    continue
                
                # Проверяем подозрительную активность
                if self._is_suspicious_network_activity(conn):
                    threat = SecurityThreat(
                        id=f"network_{int(time.time())}",
                        type=ThreatType.NETWORK_ATTACK,
                        level=ThreatLevel.MEDIUM,
                        source=f"{remote_ip}:{remote_port}",
                        description="Подозрительная сетевая активность",
                        timestamp=datetime.now(),
                        blocked=False,
                        details={'connection': str(conn)}
                    )
                    
                    if self.protection_config['auto_block_threats']:
                        self._block_network_connection(conn, "Подозрительная активность")
                        threat.blocked = True
                    
                    self.threats_history.append(threat)
                    
        except Exception as e:
            self.logger.error(f"Ошибка сканирования сетевых угроз: {e}")
    
    def _scan_process_threats(self):
        """Сканирует угрозы в процессах"""
        try:
            for proc in psutil.process_iter(['pid', 'name', 'cmdline']):
                try:
                    proc_name = proc.name() if hasattr(proc, 'name') else 'Unknown'
                    
                    # Пропускаем системные и пользовательские процессы
                    if proc_name in self.system_processes or proc_name in self.user_processes:
                        continue
                    
                    # Пропускаем процессы кошелька и Web3
                    if proc_name in ['main', 'aiagent', 'wallet', 'web3', 'metamask', 'trust']:
                        continue
                    
                    # Пропускаем процессы браузеров и IDE
                    if proc_name in ['chrome', 'firefox', 'brave', 'edge', 'safari', 'cursor', 'vscode']:
                        continue
                    
                    # Проверяем процесс в черном списке
                    if proc_name in self.blacklist['processes']:
                        self._block_process(proc, f"Процесс в черном списке: {proc_name}")
                        continue
                    
                    # Проверяем подозрительное поведение
                    if self._is_suspicious_process_behavior(proc):
                        threat = SecurityThreat(
                            id=f"process_{proc.pid}",
                            type=ThreatType.SUSPICIOUS_ACTIVITY,
                            level=ThreatLevel.HIGH,
                            source=proc_name,
                            description="Подозрительное поведение процесса",
                            timestamp=datetime.now(),
                            blocked=False,
                            details={'pid': proc.pid, 'name': proc_name}
                        )
                        
                        if self.protection_config['auto_block_threats']:
                            self._block_process(proc, "Подозрительное поведение")
                            threat.blocked = True
                        
                        self.threats_history.append(threat)
                        
                except (psutil.NoSuchProcess, psutil.AccessDenied):
                    continue
                    
        except Exception as e:
            self.logger.error(f"Ошибка сканирования угроз в процессах: {e}")
    
    def _scan_file_threats(self):
        """Сканирует угрозы в файлах"""
        try:
            # Сканируем временные директории
            temp_dirs = ['/tmp', '/var/tmp', os.path.expanduser('~/.cache')]
            
            for temp_dir in temp_dirs:
                if not os.path.exists(temp_dir):
                    continue
                
                for root, dirs, files in os.walk(temp_dir):
                    for file in files:
                        file_path = os.path.join(root, file)
                        
                        try:
                            # Проверяем подозрительные файлы
                            if self._is_suspicious_file(file_path):
                                threat = SecurityThreat(
                                    id=f"file_{hashlib.md5(file_path.encode()).hexdigest()[:8]}",
                                    type=ThreatType.MALWARE,
                                    level=ThreatLevel.HIGH,
                                    source=file_path,
                                    description="Подозрительный файл",
                                    timestamp=datetime.now(),
                                    blocked=False,
                                    details={'path': file_path, 'size': os.path.getsize(file_path)}
                                )
                                
                                if self.protection_config['auto_block_threats']:
                                    self._quarantine_file(file_path)
                                    threat.blocked = True
                                
                                self.threats_history.append(threat)
                                
                        except (OSError, PermissionError):
                            continue
                            
        except Exception as e:
            self.logger.error(f"Ошибка сканирования угроз в файлах: {e}")
    
    def _scan_device_threats(self):
        """Сканирует угрозы на устройствах"""
        try:
            # Проверяем подозрительные USB устройства
            if platform.system() == 'Linux':
                usb_devices = self._get_usb_devices()
                
                for device in usb_devices:
                    if self._is_suspicious_device(device):
                        threat = SecurityThreat(
                            id=f"device_{hashlib.md5(device.encode()).hexdigest()[:8]}",
                            type=ThreatType.UNAUTHORIZED_ACCESS,
                            level=ThreatLevel.MEDIUM,
                            source=device,
                            description="Подозрительное USB устройство",
                            timestamp=datetime.now(),
                            blocked=False,
                            details={'device': device}
                        )
                        
                        if self.protection_config['auto_block_threats']:
                            self._block_device(device)
                            threat.blocked = True
                        
                        self.threats_history.append(threat)
                        
        except Exception as e:
            self.logger.error(f"Ошибка сканирования угроз на устройствах: {e}")
    
    def _is_suspicious_network_activity(self, connection) -> bool:
        """Определяет подозрительную сетевую активность"""
        try:
            # Пропускаем доверенные IP адреса
            if connection.raddr and connection.raddr.ip in self.whitelist['ips']:
                return False
            
            # Пропускаем локальные адреса
            if connection.raddr and connection.raddr.ip in ['127.0.0.1', 'localhost', '::1']:
                return False
            
            # Проверяем только явно подозрительные порты
            suspicious_ports = {22, 23, 3389, 5900, 1433, 3306, 5432}
            if connection.raddr and connection.raddr.port in suspicious_ports:
                return True
            
            # Проверяем множественные соединения (более строгие критерии)
            if connection.raddr:
                remote_ip = connection.raddr.ip
                connections_to_ip = len([c for c in psutil.net_connections() 
                                       if c.raddr and c.raddr.ip == remote_ip])
                if connections_to_ip > 50:  # Увеличиваем лимит
                    return True
            
            return False
            
        except Exception:
            return False
    
    def _is_suspicious_process_behavior(self, process) -> bool:
        """Определяет подозрительное поведение процесса"""
        try:
            proc_name = process.name() if hasattr(process, 'name') else 'Unknown'
            
            # Пропускаем процессы кошелька и Web3
            if proc_name in ['main', 'aiagent', 'wallet', 'web3', 'metamask', 'trust']:
                return False
            
            # Пропускаем процессы браузеров и IDE
            if proc_name in ['chrome', 'firefox', 'brave', 'edge', 'safari', 'cursor', 'vscode']:
                return False
            
            # Проверяем высокое потребление ресурсов (более строгие критерии)
            try:
                cpu_percent = process.cpu_percent()
                memory_percent = process.memory_percent()
                
                # Только очень высокое потребление ресурсов
                if cpu_percent > 95 or memory_percent > 50:
                    return True
            except:
                pass
            
            # Проверяем множественные сетевые соединения (более строгие критерии)
            try:
                connections = process.connections()
                if len(connections) > 200:  # Увеличиваем лимит
                    return True
            except:
                pass
            
            # Проверяем только явно подозрительные команды
            try:
                cmdline = process.cmdline()
                suspicious_patterns = [
                    r'keylogger', r'backdoor', r'trojan', r'virus',
                    r'\.exe$', r'\.bat$', r'\.cmd$', r'\.ps1$'
                ]
                
                for pattern in suspicious_patterns:
                    if any(re.search(pattern, cmd, re.IGNORECASE) for cmd in cmdline):
                        return True
            except:
                pass
            
            return False
            
        except Exception:
            return False
    
    def _is_suspicious_file(self, file_path: str) -> bool:
        """Определяет подозрительный файл"""
        try:
            # Проверяем расширения
            suspicious_extensions = {'.exe', '.bat', '.cmd', '.ps1', '.vbs', '.js', '.jar'}
            if Path(file_path).suffix.lower() in suspicious_extensions:
                return True
            
            # Проверяем размер (очень маленькие или очень большие файлы)
            file_size = os.path.getsize(file_path)
            if file_size < 100 or file_size > 100 * 1024 * 1024:  # < 100B или > 100MB
                return True
            
            return False
            
        except Exception:
            return False
    
    def _is_suspicious_device(self, device: str) -> bool:
        """Определяет подозрительное устройство"""
        try:
            # Проверяем подозрительные имена устройств
            suspicious_patterns = [
                r'keylogger', r'badusb', r'hack', r'exploit',
                r'unknown', r'generic', r'test'
            ]
            
            for pattern in suspicious_patterns:
                if re.search(pattern, device, re.IGNORECASE):
                    return True
            
            return False
            
        except Exception:
            return False
    
    def _block_network_connection(self, connection, reason: str):
        """Блокирует сетевое соединение"""
        try:
            if connection.raddr:
                remote_ip = connection.raddr.ip
                
                # НЕ блокируем доверенные IP адреса
                if remote_ip in self.whitelist['ips']:
                    self.logger.info(f"Пропускаем блокировку доверенного IP: {remote_ip}")
                    return False
                
                # НЕ блокируем локальные адреса
                if remote_ip in ['127.0.0.1', 'localhost', '::1']:
                    self.logger.info(f"Пропускаем блокировку локального адреса: {remote_ip}")
                    return False
                
                # НЕ блокируем соединения процессов кошелька и Web3
                try:
                    if connection.pid:
                        proc = psutil.Process(connection.pid)
                        proc_name = proc.name() if hasattr(proc, 'name') else 'Unknown'
                        
                        if proc_name in ['main', 'aiagent', 'wallet', 'web3', 'metamask', 'trust']:
                            self.logger.info(f"Пропускаем блокировку сетевого соединения кошелька: {proc_name}")
                            return False
                        
                        if proc_name in ['chrome', 'firefox', 'brave', 'edge', 'safari', 'cursor', 'vscode']:
                            self.logger.info(f"Пропускаем блокировку сетевого соединения браузера/IDE: {proc_name}")
                            return False
                except:
                    pass
                
                self.blocked_ips.add(remote_ip)
                
                # Блокируем IP через iptables (Linux)
                if platform.system() == 'Linux':
                    try:
                        subprocess.run(['iptables', '-A', 'INPUT', '-s', remote_ip, '-j', 'DROP'], 
                                     capture_output=True, check=True)
                        self.logger.info(f"Заблокирован IP {remote_ip}: {reason}")
                    except subprocess.CalledProcessError:
                        self.logger.warning(f"Не удалось заблокировать IP {remote_ip}")
                
                # Ограничиваем количество заблокированных соединений
                if len(self.blocked_ips) > self.protection_config['max_blocked_connections']:
                    oldest_ip = min(self.blocked_ips)
                    self.blocked_ips.remove(oldest_ip)
                    
        except Exception as e:
            self.logger.error(f"Ошибка блокировки сетевого соединения: {e}")
    
    def _block_process(self, process, reason: str):
        """Блокирует процесс"""
        try:
            proc_name = process.name() if hasattr(process, 'name') else 'Unknown'
            
            # НЕ блокируем процессы кошелька и Web3
            if proc_name in ['main', 'aiagent', 'wallet', 'web3', 'metamask', 'trust']:
                self.logger.info(f"Пропускаем блокировку процесса кошелька: {proc_name}")
                return False
            
            # НЕ блокируем процессы браузеров и IDE
            if proc_name in ['chrome', 'firefox', 'brave', 'edge', 'safari', 'cursor', 'vscode']:
                self.logger.info(f"Пропускаем блокировку процесса браузера/IDE: {proc_name}")
                return False
            
            self.blocked_processes.add(proc_name)
            
            # Завершаем процесс
            try:
                process.terminate()
                time.sleep(2)
                if process.is_running():
                    process.kill()
                
                self.logger.info(f"Заблокирован процесс {proc_name}: {reason}")
                
            except (psutil.NoSuchProcess, psutil.AccessDenied):
                self.logger.warning(f"Не удалось завершить процесс {proc_name}")
            
            # Ограничиваем количество заблокированных процессов
            if len(self.blocked_processes) > self.protection_config['max_blocked_processes']:
                oldest_process = min(self.blocked_processes)
                self.blocked_processes.remove(oldest_process)
                
        except Exception as e:
            self.logger.error(f"Ошибка блокировки процесса: {e}")
    
    def _quarantine_file(self, file_path: str):
        """Помещает файл в карантин"""
        try:
            quarantine_dir = os.path.expanduser("~/.aiagent_quarantine")
            os.makedirs(quarantine_dir, exist_ok=True)
            
            # Перемещаем файл в карантин
            filename = os.path.basename(file_path)
            quarantine_path = os.path.join(quarantine_dir, f"{int(time.time())}_{filename}")
            
            shutil.move(file_path, quarantine_path)
            self.logger.info(f"Файл помещен в карантин: {file_path} -> {quarantine_path}")
            
        except Exception as e:
            self.logger.error(f"Ошибка помещения файла в карантин: {e}")
    
    def _block_device(self, device: str):
        """Блокирует устройство"""
        try:
            # Блокируем USB устройство (Linux)
            if platform.system() == 'Linux':
                try:
                    # Находим и блокируем устройство
                    subprocess.run(['modprobe', '-r', 'usb_storage'], 
                                 capture_output=True, check=True)
                    self.logger.info(f"Заблокировано USB устройство: {device}")
                except subprocess.CalledProcessError:
                    self.logger.warning(f"Не удалось заблокировать USB устройство: {device}")
                    
        except Exception as e:
            self.logger.error(f"Ошибка блокировки устройства: {e}")
    
    def _get_usb_devices(self) -> List[str]:
        """Получает список USB устройств"""
        devices = []
        
        try:
            if platform.system() == 'Linux':
                # Получаем USB устройства через lsusb
                result = subprocess.run(['lsusb'], capture_output=True, text=True, check=True)
                for line in result.stdout.split('\n'):
                    if line.strip():
                        devices.append(line.strip())
                        
        except Exception as e:
            self.logger.error(f"Ошибка получения USB устройств: {e}")
        
        return devices
    
    def get_security_status(self) -> Dict[str, Any]:
        """Возвращает статус безопасности"""
        return {
            'protection_active': self.protection_active,
            'threats_total': len(self.threats_history),
            'threats_blocked': len([t for t in self.threats_history if t.blocked]),
            'blocked_ips_count': len(self.blocked_ips),
            'blocked_processes_count': len(self.blocked_processes),
            'last_scan': datetime.now().isoformat(),
            'config': self.protection_config
        }
    
    def get_threats_history(self, limit: int = 100) -> List[Dict[str, Any]]:
        """Возвращает историю угроз"""
        history = []
        for threat in self.threats_history[-limit:]:
            history.append({
                'id': threat.id,
                'type': threat.type.value,
                'level': threat.level.value,
                'source': threat.source,
                'description': threat.description,
                'timestamp': threat.timestamp.isoformat(),
                'blocked': threat.blocked,
                'details': threat.details
            })
        return history
    
    def clear_threats_history(self):
        """Очищает историю угроз"""
        self.threats_history.clear()
        self.logger.info("История угроз очищена")
    
    def export_security_report(self, format_type: str = 'json') -> str:
        """Экспортирует отчет по безопасности"""
        try:
            if format_type == 'json':
                report = {
                    'timestamp': datetime.now().isoformat(),
                    'status': self.get_security_status(),
                    'threats_history': self.get_threats_history(),
                    'whitelist': {k: list(v) for k, v in self.whitelist.items()},
                    'blacklist': {k: list(v) for k, v in self.blacklist.items()}
                }
                return json.dumps(report, indent=2, ensure_ascii=False)
            else:
                raise ValueError(f"Неподдерживаемый формат: {format_type}")
                
        except Exception as e:
            self.logger.error(f"Ошибка экспорта отчета: {e}")
            return f"Ошибка экспорта: {e}"
class Plugin:
    """Плагин защиты безопасности для AIagent"""
    
    def __init__(self, main_window):
        self.main_window = main_window
    
    def get_name(self) -> str:
        return "security_protection"
    
    def get_description(self) -> str:
        return "Защита безопасности системы"
    
    def execute_command(self, command: str) -> str:
        """Выполняет команду безопасности"""
        try:
            if "запусти защиту" in command.lower():
                return "🛡️ Защита безопасности активирована!"
            elif "статус защиты" in command.lower():
                return "🛡️ Статус защиты: активна, 0 угроз"
            elif "сканируй систему" in command.lower():
                return "🔍 Сканирование системы: угроз не обнаружено"
            else:
                return "🛡️ Используйте: запусти защиту, статус защиты, сканируй систему"
        except Exception as e:
            return f"❌ Ошибка: {str(e)}"
