import requests
import json
import os
from PyQt6.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, 
    QFrame, QScrollArea, QTabWidget, QListWidget, QListWidgetItem,
    QSizePolicy, QSpacerItem, QScrollBar, QLineEdit, QComboBox,
    QSplitter, QStackedWidget, QMessageBox
)
from PyQt6.QtCore import Qt, QTimer, pyqtSignal, QSize, QThread
from PyQt6.QtGui import QFont, QColor, QPalette, QPixmap, QIcon
from typing import Dict, List, Any
import time
import traceback
from plugins.balance_manager import BalanceManager
from plugins.transaction_scanner import TransactionScanner
import webbrowser
from web3 import Web3
from config.settings import settings_manager

class TransactionLoadThread(QThread):
    """Поток для асинхронной загрузки транзакций"""
    finished = pyqtSignal(list)  # Сигнал с результатом
    error = pyqtSignal(str)  # Сигнал ошибки
    
    def __init__(self, scanner, address: str, network: str):
        super().__init__()
        self.scanner = scanner
        self.address = address
        self.network = network
    
    def run(self):
        """Выполняется в отдельном потоке"""
        try:
            print(f"🧵 [Thread] Начинаю загрузку транзакций для {self.address[:10]}... в {self.network}")
            history = self.scanner.get_transaction_history(self.address, self.network)
            print(f"🧵 [Thread] Загрузил {len(history) if history else 0} транзакций")
            self.finished.emit(history if history else [])
        except Exception as e:
            print(f"🧵 [Thread] Ошибка: {e}")
            self.error.emit(str(e))

class DataLoaderThread(QThread):
    """Поток для асинхронной загрузки токенов и NFT"""
    finished = pyqtSignal(dict)  # Сигнал с результатом
    error = pyqtSignal(str)  # Сигнал ошибки
    
    def __init__(self, balance_manager, address: str, network: str, load_nfts: bool = False):
        super().__init__()
        self.balance_manager = balance_manager
        self.address = address
        self.network = network
        self.load_nfts = load_nfts
    
    def run(self):
        """Выполняется в отдельном потоке"""
        try:
            print(f"🧵 [DataLoaderThread] Начинаю загрузку данных для {self.address[:10]}... в {self.network}")
            
            if self.load_nfts:
                # Загружаем NFT
                print(f"🧵 [DataLoaderThread] Загружаю NFT...")
                nft_data = self.balance_manager.get_nfts(self.address, self.network)
                print(f"🧵 [DataLoaderThread] NFT загружены: {len(nft_data.get('nfts', {}))} коллекций")
                self.finished.emit({"type": "nft", "data": nft_data})
            else:
                # Загружаем токены и балансы
                print(f"🧵 [DataLoaderThread] Загружаю балансы токенов...")
                balances = self.balance_manager.get_balances_from_all_tokens(self.address, self.network)
                print(f"🧵 [DataLoaderThread] Балансы загружены: {len(balances.get('tokens', {}))} токенов")
                self.finished.emit({"type": "tokens", "data": balances})
                
        except Exception as e:
            print(f"🧵 [DataLoaderThread] Ошибка: {e}")
            self.error.emit(str(e))

class ClickableFrame(QFrame):
    """Кастомный Frame который реагирует на клики"""
    def __init__(self, token_data, parent_wallet):
        super().__init__()
        self.token_data = token_data
        self.parent_wallet = parent_wallet
        self.token_index = None  # Будет установлено после создания
    
    def mousePressEvent(self, event):
        """Обработка клика"""
        self.parent_wallet.show_token_details(self.token_data, self.token_index)
        super().mousePressEvent(event)


class ModernWalletUI(QWidget):
    """Современный интерфейс кошелька с красивым дизайном"""
    
    def __init__(self, wallet, main_window):
        super().__init__()
        print("🔌 [ModernWalletUI] Инициализация начата...")
        self.wallet = wallet
        self.main_window = main_window
        self.tokens_data = {}
        self.current_view = 'tokens'  # Текущая активная вкладка (Токены по умолчанию)
        self._current_detail_widget = None  # Для хранения открытого окна деталей токена
        # Загружаем сохраненную сеть или используем Polygon по умолчанию
        saved_network = self.load_saved_network()
        self.current_network = saved_network if saved_network else "Polygon"
        print(f"🌐 Инициализация ModernWalletUI с сетью: {self.current_network}")
        
        # Принудительно устанавливаем Polygon, если сеть не определена
        if not self.current_network:
            self.current_network = "Polygon"
            print(f"🌐 Сеть не определена, устанавливаем Polygon по умолчанию")
        
        # Инициализируем BalanceManager для получения реальных балансов
        settings_container = main_window.settings_container if main_window and hasattr(main_window, 'settings_container') else None
        self.balance_manager = BalanceManager(self.wallet, settings_container)
        
        # Сразу переключаем balance_manager на текущую сеть
        network_map_ui_to_bm = {
            'Polygon': 'polygon',
            'Ethereum': 'ethereum',
            'BSC': 'bsc',
            'Arbitrum': 'arbitrum',
            'Optimism': 'optimism',
            'Avalanche': 'avalanche',
            'Base': 'base'
        }
        bm_network = network_map_ui_to_bm.get(self.current_network, 'polygon')
        if self.balance_manager:
            self.balance_manager.set_network(bm_network)
            print(f"🌐 BalanceManager установлен на сеть: {bm_network}")
        
        # Инициализируем TransactionScanner для получения истории транзакций
        self.tx_scanner = TransactionScanner()
        
        # Кэш цен для быстрого отображения
        self._cached_prices = {}
        self._last_balances = {}
        
        # Флаг текущего режима отображения
        # self.current_view = 'tokens'  # 'tokens' или 'activity'
        
        # Хранилище кнопок навигации для обновления их состояния
        self.nav_buttons = {}  # 'tokens': button, 'nft': button, 'activity': button
        
        # Защита от частых вызовов
        self.last_load_time = 0  # Инициализируем 0, чтобы первая загрузка прошла сразу
        self.min_load_interval = 3  # Минимум 3 секунды между загрузками (было 10)
        
        self.setup_ui()
        # НЕ загружаем токены при инициализации - делаем ленивую загрузку
        # self.load_tokens_data()  # Теперь загружается только при открытии кошелька
        self._tokens_loaded = False  # Флаг для отслеживания загрузки
        self.update_nav_buttons()  # Обновляем подсветку кнопок при инициализации
        
        # Таймер для обновления данных (НЕ запускаем автоматически)
        # Таймер для обновления цен (каждые 5 минут)
        self.update_timer = QTimer()
        self.update_timer.timeout.connect(self._timer_load_tokens)
        # НЕ запускаем таймер при инициализации
        # self.update_timer.start(120000)  # Отключено для быстрой загрузки
        
        # Таймер для полной перезагрузки балансов (каждые 10 минут) - для обнаружения новых токенов
        self.full_reload_timer = QTimer()
        self.full_reload_timer.timeout.connect(self._timer_full_reload)
    
    def _timer_load_tokens(self):
        """Обновление цен токенов по таймеру - только если панель видима"""
        if self.isVisible():
            print("⏰ Таймер: обновляю цены токенов (без перезагрузки списка)...")
            # Обновляем только цены, не перезагружая весь список
            self._update_prices_only()
        else:
            print("⏰ Таймер: панель скрыта, пропускаю обновление цен")
    
    def _timer_full_reload(self):
        """Полная перезагрузка балансов по таймеру - для обнаружения новых токенов"""
        if self.isVisible():
            print("⏰ Таймер полной перезагрузки: загружаю балансы для обнаружения новых токенов...")
            # Полная перезагрузка балансов - обнаружит новые токены
            self.load_tokens_data()
        else:
            print("⏰ Таймер полной перезагрузки: панель скрыта, пропускаю")
    
    def _update_prices_only(self):
        """Обновляет только цены токенов без перезагрузки всего списка"""
        try:
            # Проверяем, есть ли уже загруженные балансы
            if not hasattr(self, '_last_balances') or not self._last_balances:
                print("⚠️ Нет сохраненных балансов, пропускаю обновление цен")
                return
            
            # Получаем список символов токенов из текущего UI
            symbols = []
            balances = self._last_balances
            
            # Добавляем нативный токен
            native_token = balances.get('native_token')
            if native_token and isinstance(native_token, dict):
                symbol = native_token.get('symbol', '')
                if symbol:
                    symbols.append(symbol)
            
            # Добавляем ERC20 токены
            tokens = balances.get('tokens', {})
            if tokens and isinstance(tokens, dict):
                for symbol in tokens.keys():
                    if symbol and symbol not in symbols:
                        symbols.append(symbol)
            
            if not symbols:
                print("⚠️ Нет токенов для обновления цен")
                return
            
            print(f"🔄 Обновляю цены для {len(symbols)} токенов (без перезагрузки списка)...")
            
            # Устанавливаем флаг для мягкого обновления
            self._updating_prices_only = True
            
            # Загружаем цены асинхронно
            from threading import Thread
            prices_thread = Thread(
                target=self._load_prices_async,
                args=(symbols[:15],),  # Максимум 15 токенов
                daemon=True
            )
            prices_thread.start()
            
        except Exception as e:
            print(f"❌ Ошибка обновления цен: {e}")
            import traceback
            traceback.print_exc()
            self._updating_prices_only = False
    
    def _update_existing_widgets_prices(self, tokens_data):
        """Обновляет только цены в существующих виджетах токенов без пересоздания списка"""
        try:
            print(f"🔄 Обновляю цены в существующих виджетах для {len(tokens_data)} токенов...")
            
            # Создаем словарь для быстрого поиска по символу
            tokens_dict = {token['symbol']: token for token in tokens_data}
            
            # Проходим по всем виджетам в layout
            updated_count = 0
            for i in range(self.tokens_layout.count()):
                item = self.tokens_layout.itemAt(i)
                if not item:
                    continue
                
                widget = item.widget()
                if not widget:
                    continue
                
                # Пытаемся найти виджет токена (ClickableFrame)
                # Ищем дочерние виджеты с информацией о токене
                token_frame = widget
                if hasattr(token_frame, 'children'):
                    # Ищем QVBoxLayout с информацией о токене
                    for child in token_frame.findChildren(QVBoxLayout):
                        # Ищем QLabel с символом токена
                        for label in child.findChildren(QLabel):
                            text = label.text()
                            if text in tokens_dict:
                                token_data = tokens_dict[text]
                                # Обновляем цены в этом виджете
                                self._update_widget_prices(token_frame, token_data)
                                updated_count += 1
                                break
                
                # Альтернативный способ: ищем по сохраненным данным в frame
                if hasattr(token_frame, 'token_data'):
                    symbol = token_frame.token_data.get('symbol', '')
                    if symbol in tokens_dict:
                        token_data = tokens_dict[symbol]
                        self._update_widget_prices(token_frame, token_data)
                        updated_count += 1
            
            print(f"✅ Обновлено цен в {updated_count} виджетах")
            
            # Если не удалось обновить через виджеты, делаем полное обновление
            if updated_count == 0:
                print("⚠️ Не удалось обновить через виджеты, делаю полное обновление")
                self.update_token_list(tokens_data)
            
        except Exception as e:
            print(f"❌ Ошибка обновления виджетов: {e}")
            import traceback
            traceback.print_exc()
            # В случае ошибки делаем полное обновление
            self.update_token_list(tokens_data)
    
    def _update_widget_prices(self, token_frame, token_data):
        """Обновляет цены в конкретном виджете токена"""
        try:
            # Ищем все QLabel в виджете
            labels = token_frame.findChildren(QLabel)
            
            for label in labels:
                text = label.text()
                style = label.styleSheet()
                
                # Обновляем цену (формат: $0.0000)
                if '$' in text or 'Загрузка' in text:
                    if 'font-size: 10px' in style and 'color: #a0a0a0' in style:
                        label.setText(token_data.get('price_usd', '$0.0000'))
                
                # Обновляем изменение цены (формат: +0.00%)
                elif '%' in text or '—' in text:
                    if 'font-size: 10px' in style and ('#00b894' in style or '#e74c3c' in style or '#a0a0a0' in style):
                        label.setText(token_data.get('change', '—'))
                        label.setStyleSheet(f"font-size: 10px; color: {token_data.get('change_color', '#a0a0a0')};")
                
                # Обновляем USD значение (формат: $0.00)
                elif '$' in text and 'font-size: 9px' in style:
                    label.setText(token_data.get('usd_value', '$0.00'))
            
        except Exception as e:
            print(f"❌ Ошибка обновления виджета: {e}")

    def showEvent(self, event):
        """Вызывается когда виджет становится видимым"""
        super().showEvent(event)
        # При первом показе запускаем таймер обновления (но не загружаем сразу!)
        if not self._tokens_loaded:
            print("✅ Панель кошелька открыта - запускаю таймер обновления (через 10 сек)")
            # Первая загрузка через 10 секунд после открытия панели
            QTimer.singleShot(10000, self.load_tokens_data)
            # Запускаем периодическое обновление цен каждые 5 минут (без перезагрузки списка)
            self.update_timer.start(300000)  # 5 минут
            # Запускаем полную перезагрузку балансов каждые 10 минут (для обнаружения новых токенов)
            self.full_reload_timer.start(600000)  # 10 минут
            self._tokens_loaded = True

    def switch_network(self, network: str):
        """Переключает текущую сеть (например, 'Polygon' или 'Ethereum'),
        синхронизирует BalanceManager и сохраняет выбор.
        """
        try:
            allowed_networks = {
                'Polygon': 'polygon',
                'Ethereum': 'ethereum',
                'BSC': 'bsc',
                'Arbitrum': 'arbitrum',
                'Optimism': 'optimism',
                'Avalanche': 'avalanche',
                'Base': 'base',
            }

            if network not in allowed_networks:
                print(f"⚠️ Неизвестная сеть: {network}. Допустимые: {list(allowed_networks.keys())}")
                return

            # Обновляем текущее состояние UI
            self.current_network = network
            print(f"🌐 Переключаем сеть на: {self.current_network}")
            self.save_network_state(network)
            
            # Обновляем визуальное состояние кнопок сетей
            if hasattr(self, 'network_buttons'):
                for net_name, button in self.network_buttons.items():
                    if net_name == network:
                        # Активируем выбранную кнопку
                        button.setProperty("active", "true")
                    else:
                        # Деактивируем остальные кнопки
                        button.setProperty("active", "false")
                    # Принудительно обновляем стиль кнопки
                    button.style().unpolish(button)
                    button.style().polish(button)
                    button.update()

            # Синхронизируем BalanceManager
            bm_network = allowed_networks[network]
            if self.balance_manager:
                self.balance_manager.set_network(bm_network)
                print(f"🌐 BalanceManager переключен на: {bm_network}")

            # Перезагружаем данные токенов
            self.load_tokens_data()
            
            # Обновляем сеть в SwapPanel, если он открыт
            if hasattr(self, 'exchange_form_widget') and self.exchange_form_widget.isVisible():
                self.exchange_form_widget.update_network(network)
                print(f"🌐 SwapPanel обновлен на сеть: {network}")
            
            # 🆕 ИСПРАВЛЕНИЕ: Если в данный момент открыта вкладка Activity,
            # перезагружаем историю транзакций для новой сети
            if self.current_view == 'activity':
                print(f"🌐 Activity открыта, перезагружаю историю для новой сети: {self.current_network}")
                self.show_activity()
            
        except Exception as e:
            print(f"❌ Ошибка переключения сети: {e}")
    
    def save_network_state(self, network):
        """Сохраняет выбранную сеть в файл"""
        try:
            import json
            import os
            
            # Создаем директорию если не существует
            config_dir = "config/wallets"
            os.makedirs(config_dir, exist_ok=True)
            
            # Сохраняем состояние сети
            state_file = os.path.join(config_dir, "network_state.json")
            state_data = {
                "current_network": network,
                "timestamp": time.time()
            }
            
            with open(state_file, 'w', encoding='utf-8') as f:
                json.dump(state_data, f, ensure_ascii=False, indent=2)
            
            print(f"💾 Сохранена выбранная сеть: {network}")
        except Exception as e:
            print(f"❌ Ошибка сохранения состояния сети: {e}")
    
    def load_saved_network(self):
        """Загружает сохраненную сеть из файла"""
        try:
            import json
            import os
            
            state_file = "config/wallets/network_state.json"
            if os.path.exists(state_file):
                with open(state_file, 'r', encoding='utf-8') as f:
                    state_data = json.load(f)
                
                # Проверяем, что файл не слишком старый (не старше 7 дней)
                if time.time() - state_data.get('timestamp', 0) < 7 * 24 * 3600:
                    network = state_data.get('current_network')
                    if network:
                        print(f"📂 Загружена сохраненная сеть: {network}")
                        return network
            
            print("📂 Сохраненная сеть не найдена, используем Polygon по умолчанию")
            return None
        except Exception as e:
            print(f"❌ Ошибка загрузки состояния сети: {e}")
            return None
    
    def update_nav_buttons(self):
        """Обновляет подсветку активной кнопки навигации"""
        try:
            # Стиль для активной кнопки (фиолетовая)
            active_style = """
                QPushButton {
                    background-color: #6c5ce7;
                    color: #ffffff;
                    border: none;
                    border-radius: 6px;
                    font-size: 11px;
                    font-weight: bold;
                    padding: 6px 12px;
                }
                QPushButton:hover {
                    background-color: #5a4fcf;
                }
            """
            
            # Стиль для неактивной кнопки (тёмная)
            inactive_style = """
                QPushButton {
                    background-color: #2c2c2c;
                    color: #ffffff;
                    border: none;
                    border-radius: 6px;
                    font-size: 11px;
                    font-weight: bold;
                    padding: 6px 12px;
                }
                QPushButton:hover {
                    background-color: #4c4c4c;
                }
            """
            
            # Обновляем стили кнопок на основе current_view
            if 'tokens' in self.nav_buttons:
                if self.current_view == 'tokens':
                    self.nav_buttons['tokens'].setStyleSheet(active_style)
                else:
                    self.nav_buttons['tokens'].setStyleSheet(inactive_style)
            
            if 'nft' in self.nav_buttons:
                if self.current_view == 'nft':
                    self.nav_buttons['nft'].setStyleSheet(active_style)
                else:
                    self.nav_buttons['nft'].setStyleSheet(inactive_style)
            
            if 'activity' in self.nav_buttons:
                if self.current_view == 'activity':
                    self.nav_buttons['activity'].setStyleSheet(active_style)
                else:
                    self.nav_buttons['activity'].setStyleSheet(inactive_style)
            
            print(f"🎨 Обновили подсветку кнопок навигации (активная: {self.current_view})")
        except Exception as e:
            print(f"❌ Ошибка обновления кнопок навигации: {e}")
    
    def setup_ui(self):
        """Настройка пользовательского интерфейса"""
        print("🔌 [setup_ui] Начинаю создание интерфейса...")
        self.setStyleSheet("""
            QWidget {
                background-color: #1a1a1a;
                color: #ffffff;
                font-family: 'Segoe UI', Arial, sans-serif;
            }
            QPushButton {
                background-color: #6c5ce7;
                color: white;
                border: none;
                border-radius: 6px;
                padding: 8px 12px;
                font-weight: bold;
                font-size: 12px;
            }
            QPushButton:hover {
                background-color: #5f3dc4;
            }
            QPushButton:pressed {
                background-color: #4c63d2;
            }
            QPushButton#exit_button {
                background-color: #e74c3c;
                padding: 4px 8px;
                font-size: 10px;
            }
            QPushButton#exit_button:hover {
                background-color: #c0392b;
            }
            QPushButton#network_button {
                background-color: #2c2c2c;
                padding: 4px 8px;
                font-size: 10px;
                border-radius: 4px;
            }
            QPushButton#network_button:hover {
                background-color: #3c3c3c;
            }
            QPushButton#network_button.active {
                background-color: #6c5ce7;
            }
            QPushButton#add_token_button {
                background-color: #00b894;
                padding: 12px;
                font-size: 14px;
                border-radius: 8px;
            }
            QPushButton#add_token_button:hover {
                background-color: #00a085;
            }
            QLabel {
                color: #ffffff;
            }
            QFrame {
                background-color: #2c2c2c;
                border-radius: 12px;
                border: 1px solid #3c3c3c;
            }
            QTabWidget::pane {
                border: none;
                background-color: transparent;
            }
            QTabBar::tab {
                background-color: #2c2c2c;
                color: #ffffff;
                padding: 12px 20px;
                margin-right: 2px;
                border-radius: 8px 8px 0 0;
            }
            QTabBar::tab:selected {
                background-color: #6c5ce7;
            }
            QTabBar::tab:hover {
                background-color: #3c3c3c;
            }
        """)
        
        # Основной layout
        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(10, 10, 10, 10)
        main_layout.setSpacing(8)
        
        # Заголовок с адресом и кнопкой выхода
        self.create_header(main_layout)
        
        # Основной баланс MATIC
        self.create_main_balance(main_layout)
        
        # Кнопки действий
        self.create_action_buttons(main_layout)
        
        # Фильтры сетей (под кнопками действий)
        self.create_network_filters(main_layout)
        
        # Формы действий (обмен, отправка, получение) - раскрываются под блоком выбора сети
        self.create_action_forms(main_layout)
        
        # Вкладки навигации
        self.create_navigation_tabs(main_layout)
        
        # Список токенов
        self.create_token_list(main_layout)
        
        # Кнопка добавления токенов
        self.create_add_token_button(main_layout)
    
    def create_header(self, parent_layout):
        """Создание заголовка с адресом и кнопкой выхода"""
        header_frame = QFrame()
        header_frame.setFixedHeight(40)
        header_layout = QHBoxLayout(header_frame)
        header_layout.setContentsMargins(5, 5, 5, 5)
        
        # Адрес кошелька
        self.address_label = QLabel("Адрес: Загрузка...")
        self.address_label.setStyleSheet("font-size: 12px; font-weight: bold;")
        header_layout.addWidget(self.address_label)
        
        # Кнопка обновить кошелек
        refresh_button = QPushButton("⟳")
        refresh_button.setFixedSize(30, 30)  # Квадратная кнопка
        refresh_button.setToolTip("Обновить баланс кошелька")
        refresh_button.clicked.connect(self.load_tokens_data)
        refresh_button.setStyleSheet("""
            QPushButton {
                background-color: #2c2c2c;
                color: #ffffff;
                border: 1px solid #3c3c3c;
                border-radius: 4px;
                font-size: 18px;
                padding: 0;
            }
            QPushButton:hover {
                background-color: #3c3c3c;
                border-color: #6c5ce7;
            }
        """)
        header_layout.addWidget(refresh_button)
        
        # Кнопка выхода
        exit_button = QPushButton("Выход")
        exit_button.setObjectName("exit_button")
        exit_button.setFixedSize(60, 30)
        exit_button.clicked.connect(self.exit_wallet)
        header_layout.addWidget(exit_button)
        
        parent_layout.addWidget(header_frame)
    
    def create_main_balance(self, parent_layout):
        """Создание секции с основным балансом нативного токена"""
        balance_frame = QFrame()
        balance_frame.setFixedHeight(60)
        balance_frame.setStyleSheet("""
            QFrame {
                background-color: #2c2c2c;
                border-radius: 8px;
                border: none;
            }
        """)
        
        balance_layout = QVBoxLayout(balance_frame)
        balance_layout.setAlignment(Qt.AlignmentFlag.AlignCenter)
        balance_layout.setContentsMargins(5, 5, 5, 5)
        
        # Основной баланс (будет обновляться динамически)
        self.main_balance_label = QLabel("Загрузка...")
        self.main_balance_label.setStyleSheet("""
            font-size: 18px;
            font-weight: bold;
            color: #ffffff;
        """)
        self.main_balance_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        balance_layout.addWidget(self.main_balance_label)
        
        # USD эквивалент (будет обновляться динамически)
        self.usd_balance_label = QLabel("≈ $0.00 USD")
        self.usd_balance_label.setStyleSheet("""
            font-size: 12px;
            color: #a0a0a0;
        """)
        self.usd_balance_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        balance_layout.addWidget(self.usd_balance_label)
        
        parent_layout.addWidget(balance_frame)
    
    def create_action_buttons(self, parent_layout):
        """Создание кнопок действий"""
        buttons_layout = QHBoxLayout()
        buttons_layout.setSpacing(5)
        
        # Кнопка "Обмен"
        collect_button = QPushButton("Обмен")
        collect_button.setFixedHeight(35)
        collect_button.clicked.connect(self.toggle_exchange_form)
        buttons_layout.addWidget(collect_button)
        
        # Кнопка "Отправить"
        send_button = QPushButton("Отправить")
        send_button.setFixedHeight(35)
        send_button.clicked.connect(self.toggle_send_form)
        buttons_layout.addWidget(send_button)
        
        # Кнопка "Получить"
        receive_button = QPushButton("Получить")
        receive_button.setFixedHeight(35)
        receive_button.clicked.connect(self.toggle_receive_form)
        buttons_layout.addWidget(receive_button)
        
        parent_layout.addLayout(buttons_layout)
        
    def create_action_forms(self, parent_layout):
        """Создание раскрывающихся форм (вызывается после network_filters)"""
        # Создаем раскрывающиеся формы
        self.exchange_form_widget = self.create_exchange_form()
        self.exchange_form_widget.setVisible(False)  # Изначально скрыта
        parent_layout.addWidget(self.exchange_form_widget)
        
        self.send_form_widget = self.create_send_form()
        self.send_form_widget.setVisible(False)  # Изначально скрыта
        parent_layout.addWidget(self.send_form_widget)
        
        self.receive_form_widget = self.create_receive_form()
        self.receive_form_widget.setVisible(False)  # Изначально скрыта
        parent_layout.addWidget(self.receive_form_widget)
    
    def create_navigation_tabs(self, parent_layout):
        """Создание кнопок навигации без содержимого"""
        # Создаем виджет-контейнер для кнопок навигации
        self.tabs_widget = QWidget()
        buttons_layout = QHBoxLayout(self.tabs_widget)
        buttons_layout.setSpacing(5)
        buttons_layout.setContentsMargins(0, 0, 0, 0)
        
        # Кнопка "Токены"
        tokens_button = QPushButton("Токены")
        tokens_button.setFixedHeight(30)
        tokens_button.setStyleSheet("""
            QPushButton {
                background-color: #6c5ce7;
                color: #ffffff;
                border: none;
                border-radius: 6px;
                font-size: 11px;
                font-weight: bold;
                padding: 6px 12px;
            }
            QPushButton:hover {
                background-color: #5a4fcf;
            }
        """)
        tokens_button.clicked.connect(self.show_tokens)
        buttons_layout.addWidget(tokens_button)
        self.nav_buttons['tokens'] = tokens_button  # Сохраняем ссылку
        
        # Кнопка "NFT"
        nft_button = QPushButton("NFT")
        nft_button.setFixedHeight(30)
        nft_button.setStyleSheet("""
            QPushButton {
                background-color: #2c2c2c;
                color: #ffffff;
                border: none;
                border-radius: 6px;
                font-size: 11px;
                font-weight: bold;
                padding: 6px 12px;
            }
            QPushButton:hover {
                background-color: #4c4c4c;
            }
        """)
        nft_button.clicked.connect(self.show_nft)
        buttons_layout.addWidget(nft_button)
        self.nav_buttons['nft'] = nft_button  # Сохраняем ссылку
        
        # Кнопка "Активность"
        activity_button = QPushButton("Активность")
        activity_button.setFixedHeight(30)
        activity_button.setStyleSheet("""
            QPushButton {
                background-color: #2c2c2c;
                color: #ffffff;
                border: none;
                border-radius: 6px;
                font-size: 11px;
                font-weight: bold;
                padding: 6px 12px;
            }
            QPushButton:hover {
                background-color: #4c4c4c;
            }
        """)
        activity_button.clicked.connect(self.show_activity)
        buttons_layout.addWidget(activity_button)
        self.nav_buttons['activity'] = activity_button  # Сохраняем ссылку
        
        parent_layout.addWidget(self.tabs_widget)
    
    def create_network_filters(self, parent_layout):
        """Создание фильтров сетей с горизонтальным скроллингом"""
        # Создаем контейнер для сетей
        networks_container = QFrame()
        networks_container.setFixedHeight(50)
        networks_container.setStyleSheet("""
            QFrame {
                background-color: #2c2c2c;
                border-radius: 8px;
                border: 1px solid #3c3c3c;
            }
        """)
        
        # Создаем горизонтальный layout для кнопок сетей
        networks_layout = QHBoxLayout(networks_container)
        networks_layout.setContentsMargins(10, 8, 10, 8)
        networks_layout.setSpacing(8)
        
        # Все поддерживаемые сети
        networks = [
            "Ethereum", "Polygon", "BSC", "Arbitrum", "Optimism", 
            "Avalanche", "Base", "Linea", "Scroll", "Mantle", 
            "Blast", "Mode"
        ]
        
        # Создаем кнопки для каждой сети
        self.network_buttons = {}
        for network in networks:
            network_button = QPushButton(network)
            network_button.setObjectName("network_button")
            network_button.setFixedSize(80, 32)
            network_button.setStyleSheet("""
                QPushButton#network_button {
                    background-color: #3c3c3c;
                    color: #ffffff;
                    border: 1px solid #4c4c4c;
                    border-radius: 6px;
                    font-size: 11px;
                    font-weight: bold;
                }
                QPushButton#network_button:hover {
                    background-color: #4c4c4c;
                    border-color: #6c5ce7;
                }
                QPushButton#network_button[active="true"] {
                    background-color: #4c4c4c;
                    border-color: #6c5ce7;
                }
            """)
            
            # Устанавливаем активную сеть на основе загруженного состояния
            if network == self.current_network:
                network_button.setProperty("active", "true")
            else:
                network_button.setProperty("active", "false")
            
            network_button.clicked.connect(lambda checked, n=network: self.switch_network(n))
            networks_layout.addWidget(network_button)
            self.network_buttons[network] = network_button
        
        # Добавляем растягивающийся элемент в конец
        networks_layout.addStretch()
        
        # Создаем горизонтальный скролл-арea
        scroll_area = QScrollArea()
        scroll_area.setWidget(networks_container)
        scroll_area.setWidgetResizable(False)
        scroll_area.setFixedHeight(50)
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAsNeeded)
        scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        scroll_area.setStyleSheet("""
            QScrollArea {
                border: none;
                background-color: transparent;
            }
            QScrollBar:horizontal {
                background-color: #2c2c2c;
                height: 8px;
                border-radius: 4px;
            }
            QScrollBar::handle:horizontal {
                background-color: #6c5ce7;
                border-radius: 4px;
                min-width: 20px;
            }
            QScrollBar::handle:horizontal:hover {
                background-color: #5f3dc4;
            }
        """)
        
        parent_layout.addWidget(scroll_area)
    
    def create_token_list(self, parent_layout):
        """Создание списка токенов"""
        # Scroll area для списка токенов
        self.token_scroll_area = QScrollArea()
        self.token_scroll_area.setWidgetResizable(True)
        self.token_scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAsNeeded)
        self.token_scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        self.token_scroll_area.setStyleSheet("""
            QScrollArea {
                border: none;
                background-color: transparent;
            }
            QScrollBar:vertical {
                background-color: #2c2c2c;
                width: 8px;
                border-radius: 4px;
            }
            QScrollBar::handle:vertical {
                background-color: #6c5ce7;
                border-radius: 4px;
            }
        """)
        
        # Контейнер для токенов
        self.tokens_container = QWidget()
        self.tokens_layout = QVBoxLayout(self.tokens_container)
        self.tokens_layout.setSpacing(8)
        self.tokens_layout.setContentsMargins(4, 4, 4, 4)  # Небольшие отступы для адаптивности
        
        # АДАПТИВ: Убеждаемся что токены не сжимаются
        self.tokens_container.setMinimumWidth(300)  # Минимальная ширина контейнера
        self.tokens_container.setSizePolicy(
            QSizePolicy.Policy.Expanding, 
            QSizePolicy.Policy.Minimum
        )
        
        self.token_scroll_area.setWidget(self.tokens_container)
        self.token_scroll_area.setWidgetResizable(True)  # Важно для адаптива!
        parent_layout.addWidget(self.token_scroll_area)
    
    def create_add_token_button(self, parent_layout):
        """Создание кнопки добавления токенов"""
        self.add_token_button = QPushButton("+ Добавить токен")
        self.add_token_button.setObjectName("add_token_button")
        self.add_token_button.setFixedHeight(25)  # Уменьшаем высоту на 30% (было ~35px)
        self.add_token_button.clicked.connect(self.toggle_add_token_form)
        self.add_token_button.setStyleSheet("""
            QPushButton {
                background-color: #7fff00;
                color: #1a1a1a;
                border: none;
                border-radius: 6px;
                font-size: 12px;
                font-weight: bold;
                padding: 5px;
            }
            QPushButton:hover {
                background-color: #66cc00;
            }
            QPushButton:pressed {
                background-color: #55aa00;
            }
        """)
        parent_layout.addWidget(self.add_token_button)
        
        # Создаем раскрывающуюся форму добавления токена
        self.add_token_form_widget = self.create_add_token_form()
        self.add_token_form_widget.setVisible(False)  # Изначально скрыта
        parent_layout.addWidget(self.add_token_form_widget)
    
    def load_tokens_data(self):
        """Загружает данные токенов асинхронно в отдельном потоке"""
        print("🔌 [load_tokens_data] Вызов начат")
        if not self.wallet or not self.wallet.account:
            print("⚠️ Кошелек не инициализирован")
            return
        
        address = self.wallet.account.address
        print(f"🔌 [load_tokens_data] Адрес: {address}")
        
        # Очищаем контейнер
        print(f"💰 Очищаю tokens_layout перед загрузкой")
        while self.tokens_layout.count():
            child = self.tokens_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        
        # Показываем индикатор загрузки
        loading_label = QLabel("⏳ Загружаю токены...")
        loading_label.setStyleSheet("color: #b0b0b0; font-size: 13px; padding: 20px;")
        loading_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.tokens_layout.addWidget(loading_label)
        
        # Запускаем загрузку в отдельном потоке
        self.data_loader_thread = DataLoaderThread(
            self.balance_manager,
            address,
            self.current_network,
            load_nfts=False
        )
        self.data_loader_thread.finished.connect(self._on_tokens_loaded)
        self.data_loader_thread.error.connect(self._on_loader_error)
        self.data_loader_thread.start()
    
    def load_nfts_data(self):
        """Загружает NFT данные асинхронно в отдельном потоке"""
        print("🎨 [load_nfts_data] Вызов начат")
        if not self.wallet or not self.wallet.account:
            print("⚠️ Кошелек не инициализирован")
            return
        
        address = self.wallet.account.address
        print(f"🎨 [load_nfts_data] Адрес: {address}")
        
        # Очищаем контейнер
        print(f"🎨 Очищаю tokens_layout перед загрузкой")
        while self.tokens_layout.count():
            child = self.tokens_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        
        # Показываем индикатор загрузки
        loading_label = QLabel("⏳ Загружаю NFT...")
        loading_label.setStyleSheet("color: #b0b0b0; font-size: 13px; padding: 20px;")
        loading_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.tokens_layout.addWidget(loading_label)
        
        # Запускаем загрузку в отдельном потоке
        self.data_loader_thread = DataLoaderThread(
            self.balance_manager,
            address,
            self.current_network,
            load_nfts=True
        )
        self.data_loader_thread.finished.connect(self._on_nfts_loaded)
        self.data_loader_thread.error.connect(self._on_loader_error)
        self.data_loader_thread.start()
    
    def _on_tokens_loaded(self, result):
        """Callback при загрузке токенов"""
        # Проверяем что это для токенов, а не для NFT
        if not isinstance(result, dict) or result.get('type') != 'tokens':
            return
        
        balances = result.get('data', {})
        
        # Обновляем время последней загрузки
        import time
        self.last_load_time = time.time()
        print(f"✅ Токены загружены, обновлен last_load_time: {self.last_load_time}")
        
        # Обновляем адрес в заголовке
        if self.wallet and self.wallet.account:
            address = self.wallet.account.address
            # Сокращаем адрес: 0x1234...5678
            short_address = f"{address[:6]}...{address[-4:]}"
            self.address_label.setText(f"Адрес: {short_address}")
            self.address_label.setToolTip(f"Полный адрес: {address}")  # Подсказка с полным адресом
        
        # Очищаем старые виджеты
        while self.tokens_layout.count():
            child = self.tokens_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        
        # Обновляем интерфейс с новыми данными
        self._update_ui_with_balances(balances)
    
    def _on_nfts_loaded(self, result):
        """Callback при загрузке NFT"""
        print("✅ NFT загружены, обновляю UI...")
        nft_data = result.get('data', {})
        nfts = nft_data.get('nfts', {})
        
        # Очищаем старые виджеты
        while self.tokens_layout.count():
            child = self.tokens_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        
        # Если нет NFT, показываем сообщение
        if not nfts:
            nft_label = QLabel("🎨 У вас нет NFT")
            nft_label.setStyleSheet("color: #a0a0a0; font-size: 14px; padding: 20px;")
            nft_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            self.tokens_layout.addWidget(nft_label)
            return
        
        # Отображаем каждый NFT
        for contract_addr, nft_info in nfts.items():
            symbol = nft_info.get('symbol', 'NFT')
            name = nft_info.get('name', 'Unknown')
            tokens = nft_info.get('tokens', [])
            
            for token in tokens:
                token_id = token.get('token_id', 'unknown')
                
                # Создаем карточку NFT
                nft_frame = QFrame()
                nft_frame.setStyleSheet("""
                    QFrame {
                        background-color: #2c2c2c;
                        border: 1px solid #3c3c3c;
                        border-radius: 8px;
                        padding: 12px;
                        margin: 4px 0px;
                    }
                    QFrame:hover {
                        background-color: #333333;
                        border-color: #6c5ce7;
                    }
                """)
                nft_frame.setCursor(Qt.CursorShape.PointingHandCursor)
                
                nft_layout = QVBoxLayout(nft_frame)
                nft_layout.setContentsMargins(12, 12, 12, 12)
                nft_layout.setSpacing(6)
                
                # Название NFT
                title_label = QLabel(f"🎨 {symbol}")
                title_font = QFont()
                title_font.setPointSize(12)
                title_font.setBold(True)
                title_label.setFont(title_font)
                title_label.setStyleSheet("color: #ffffff;")
                nft_layout.addWidget(title_label)
                
                # Имя NFT
                name_label = QLabel(f"📝 {name}")
                name_label.setStyleSheet("color: #b0b0b0; font-size: 11px;")
                nft_layout.addWidget(name_label)
                
                # ID токена
                id_label = QLabel(f"🆔 ID: {token_id}")
                id_label.setStyleSheet("color: #808080; font-size: 10px;")
                id_label.setWordWrap(True)
                nft_layout.addWidget(id_label)
                
                # Адрес контракта
                addr_label = QLabel(f"📍 {contract_addr[:12]}...{contract_addr[-10:]}")
                addr_label.setStyleSheet("color: #707070; font-size: 9px;")
                addr_label.setWordWrap(True)
                nft_layout.addWidget(addr_label)
                
                self.tokens_layout.addWidget(nft_frame)
        
        # Добавляем растяжитель вниз
        self.tokens_layout.addStretch()
    
    def _on_loader_error(self, error_msg):
        """Callback при ошибке загрузки"""
        print(f"❌ Ошибка загрузки: {error_msg}")
        
        # Очищаем старые виджеты
        while self.tokens_layout.count():
            child = self.tokens_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        
        # Показываем ошибку
        error_label = QLabel(f"❌ Ошибка загрузки данных")
        error_label.setStyleSheet("color: #ff6b6b; font-size: 12px; padding: 20px;")
        error_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.tokens_layout.addWidget(error_label)
    
    def _load_prices_async(self, symbols):
        """Асинхронная загрузка цен токенов в фоне"""
        try:
            print(f"🔄 [Асинхронно] Загружаем курсы для {len(symbols)} токенов...")
            
            # Получаем цены
            token_prices = self.get_token_prices(symbols)
            
            # Кэшируем цены (сохраняем полный dict, а не только float)
            if not hasattr(self, '_cached_prices'):
                self._cached_prices = {}
            for symbol, data in token_prices.items():
                if isinstance(data, dict):
                    self._cached_prices[symbol] = data  # Сохраняем полный dict
            
            print(f"✅ [Асинхронно] Получены курсы: {len(token_prices)} токенов")
            
            # Обновляем UI через QTimer (должен выполняться в главном потоке)
            from PyQt6.QtCore import QTimer
            
            def update_ui():
                try:
                    # Получаем сохраненные балансы
                    if not hasattr(self, '_last_balances'):
                        return
                    balances = self._last_balances
                    
                    # Пересоздаем tokens_data с актуальными ценами
                    tokens_data = []
                    
                    # Добавляем нативный токен
                    native_token = balances.get('native_token')
                    if native_token and isinstance(native_token, dict):
                        symbol = native_token.get('symbol', '')
                        balance = native_token.get('balance', 0)
                        price_data = token_prices.get(symbol, {})
                        # Безопасная обработка
                        if isinstance(price_data, dict):
                            price_usd = price_data.get('price_usd', 0) or 0
                            change_24h = price_data.get('change_24h', 0)
                        elif isinstance(price_data, (int, float)):
                            price_usd = price_data
                            change_24h = 0
                        else:
                            price_usd = native_token.get('price_usd') or 0
                            change_24h = 0
                        
                        if not price_usd and symbol in ['POL', 'MATIC']:
                            price_usd = self.get_pol_price_fallback() or 0.2
                        
                        usd_value = float(balance) * price_usd
                        
                        tokens_data.append({
                            "symbol": symbol,
                            "name": symbol,
                            "balance": f"{balance:.4f}",
                            "usd_value": f"${usd_value:.2f}" if price_usd > 0 else "$0.00",
                            "price_usd": f"${price_usd:.4f}" if price_usd > 0 else "Загрузка...",
                            "change": f"{change_24h:+.2f}%" if price_usd > 0 else "—",
                            "change_color": "#00b894" if change_24h >= 0 else "#e74c3c" if price_usd > 0 else "#a0a0a0"
                        })
                    
                    # Добавляем ERC20 токены
                    tokens = balances.get('tokens', {})
                    if tokens and isinstance(tokens, dict):
                        for symbol, token_data in tokens.items():
                            if not isinstance(token_data, dict):
                                continue
                            
                            price_data = token_prices.get(symbol, {})
                            # Безопасная обработка
                            if isinstance(price_data, dict):
                                price_usd = price_data.get('price_usd', 0) or 0
                                change_24h = price_data.get('change_24h', 0)
                            elif isinstance(price_data, (int, float)):
                                price_usd = price_data
                                change_24h = 0
                            else:
                                price_usd = 0
                                change_24h = 0
                            
                            balance = token_data.get('balance', 0)
                            usd_value = float(balance) * price_usd
                            
                            tokens_data.append({
                                "symbol": symbol,
                                "name": symbol,
                                "balance": f"{balance:.4f}",
                                "usd_value": f"${usd_value:.2f}" if price_usd > 0 else "$0.00",
                                "price_usd": f"${price_usd:.4f}" if price_usd > 0 else "Загрузка...",
                                "change": f"{change_24h:+.2f}%" if price_usd > 0 else "—",
                                "change_color": "#00b894" if change_24h >= 0 else "#e74c3c" if price_usd > 0 else "#a0a0a0"
                            })
                    
                    # Подсчитываем общую сумму в USD всех токенов
                    total_usd = 0
                    for token in tokens_data:
                        usd_str = token.get('usd_value', '$0.00')
                        # Извлекаем числовое значение из строки "$123.45"
                        try:
                            usd_value = float(usd_str.replace('$', '').replace(',', ''))
                            total_usd += usd_value
                        except:
                            pass
                    
                    # Обновляем общий баланс в USD
                    if hasattr(self, 'usd_balance_label'):
                        self.usd_balance_label.setText(f"≈ ${total_usd:,.2f} USD")
                        print(f"💰 [Асинхронно] Общий баланс обновлен: ${total_usd:,.2f} USD")
                    
                    # Обновляем список токенов только если это не обновление цен по таймеру
                    if tokens_data:
                        # Проверяем, идет ли обновление по таймеру (мягкое обновление)
                        if hasattr(self, '_updating_prices_only') and self._updating_prices_only:
                            # Обновляем только цены в существующих виджетах
                            self._update_existing_widgets_prices(tokens_data)
                            self._updating_prices_only = False
                        else:
                            # Полное обновление списка
                            self.update_token_list(tokens_data)
                        print(f"✅ [Асинхронно] UI обновлен с актуальными ценами")
                    
                except Exception as e:
                    print(f"❌ [Асинхронно] Ошибка обновления UI: {e}")
            
            # Используем QTimer для выполнения в главном потоке Qt
            QTimer.singleShot(0, update_ui)
            
        except Exception as e:
            print(f"❌ [Асинхронно] Ошибка загрузки цен: {e}")
            import traceback
            traceback.print_exc()
    
    def _update_ui_with_balances(self, balances):
        """Обновление UI с балансами (упрощенная версия)"""
        try:
            if not isinstance(balances, dict):
                return
            
            tokens_data = []
            all_symbols = []
            
            # Добавляем нативный токен
            native_token = balances.get('native_token')
            if native_token and isinstance(native_token, dict):
                symbol = native_token.get('symbol', '')
                balance = native_token.get('balance', 0)
                # Преобразуем в float если это строка
                try:
                    balance = float(balance)
                except:
                    balance = 0
                all_symbols.append(symbol)
                
                if hasattr(self, 'main_balance_label') and symbol:
                    self.main_balance_label.setText(f"{balance:.4f} {symbol}")
                if hasattr(self, 'usd_balance_label'):
                    self.usd_balance_label.setText("≈ Загрузка... USD")
                
                # Получаем цену из кэша или используем 0
                price_usd = 0
                change_24h = 0
                if hasattr(self, '_cached_prices') and symbol in self._cached_prices:
                    cached = self._cached_prices[symbol]
                    if isinstance(cached, dict):
                        price_usd = cached.get('price_usd', 0) or 0
                        change_24h = cached.get('change_24h', 0)
                    elif isinstance(cached, (int, float)):
                        price_usd = cached
                
                if not price_usd and symbol in ['POL', 'MATIC']:
                    price_usd = self.get_pol_price_fallback() or 0.2
                
                usd_value = float(balance) * price_usd
                
                tokens_data.append({
                    "symbol": symbol,
                    "name": symbol,
                    "balance": f"{balance:.4f}",
                    "usd_value": f"${usd_value:.2f}" if price_usd > 0 else "$0.00",
                    "price_usd": f"${price_usd:.4f}" if price_usd > 0 else "Загрузка...",
                    "change": f"{change_24h:+.2f}%" if price_usd > 0 else "—",
                    "change_color": "#00b894" if change_24h >= 0 else "#e74c3c" if price_usd > 0 else "#a0a0a0",
                    "address": ""  # Нативный токен не имеет адреса контракта
                })
            
            # Добавляем ERC20 токены
            tokens = balances.get('tokens', {})
            if tokens and isinstance(tokens, dict):
                for symbol, token_data in tokens.items():
                    if not isinstance(token_data, dict):
                        continue
                    
                    # Пропускаем POL из ERC20, так как уже есть нативный POL
                    if symbol in ['POL', 'MATIC']:
                        continue
                    
                    all_symbols.append(symbol)
                    balance = token_data.get('balance', 0)
                    # Преобразуем в float если это строка
                    try:
                        balance = float(balance)
                    except:
                        balance = 0
                    
                    # Получаем цену из кэша
                    price_usd = 0
                    change_24h = 0
                    if hasattr(self, '_cached_prices') and symbol in self._cached_prices:
                        cached = self._cached_prices[symbol]
                        if isinstance(cached, dict):
                            price_usd = cached.get('price_usd', 0) or 0
                            change_24h = cached.get('change_24h', 0)
                        elif isinstance(cached, (int, float)):
                            price_usd = cached
                    
                    usd_value = float(balance) * price_usd
                    
                    # Получаем адрес контракта из token_data
                    contract_address = token_data.get('address', '')
                    
                    tokens_data.append({
                        "symbol": symbol,
                        "name": symbol,
                        "balance": f"{balance:.4f}",
                        "usd_value": f"${usd_value:.2f}" if price_usd > 0 else "$0.00",
                        "price_usd": f"${price_usd:.4f}" if price_usd > 0 else "Загрузка...",
                        "change": f"{change_24h:+.2f}%" if price_usd > 0 else "—",
                        "change_color": "#00b894" if change_24h >= 0 else "#e74c3c" if price_usd > 0 else "#a0a0a0",
                        "address": contract_address  # Адрес контракта для раскрывающегося окна
                    })
            
            self._last_balances = balances
            
            if not tokens_data:
                tokens_data = [{
                    "symbol": "N/A",
                    "name": "Нет токенов",
                    "balance": "0.0000",
                    "usd_value": "$0.00",
                    "price_usd": "N/A",
                    "change": "—",
                    "change_color": "#a0a0a0",
                    "address": ""
                }]
            
            # Подсчитываем общую сумму в USD всех токенов
            total_usd = 0
            for token in tokens_data:
                usd_str = token.get('usd_value', '$0.00')
                # Извлекаем числовое значение из строки "$123.45"
                try:
                    usd_value = float(usd_str.replace('$', '').replace(',', ''))
                    total_usd += usd_value
                except:
                    pass
            
            # Обновляем общий баланс в USD
            if hasattr(self, 'usd_balance_label'):
                self.usd_balance_label.setText(f"≈ ${total_usd:,.2f} USD")
                print(f"💰 Общий баланс: ${total_usd:,.2f} USD")
            
            self.update_token_list(tokens_data)
            
            # Запускаем асинхронную загрузку цен для токенов с балансом
            if all_symbols:
                from threading import Thread
                tokens_to_price = all_symbols[:15]  # Максимум 15 токенов
                prices_thread = Thread(
                    target=self._load_prices_async,
                    args=(tokens_to_price,),
                    daemon=True
                )
                prices_thread.start()
                
        except Exception as e:
            import traceback
            print(f"❌ Ошибка в _update_ui_with_balances: {e}")
            print(f"📋 Traceback: {''.join(traceback.format_exc())}")
    
    def update_token_list(self, tokens_data):
        """Обновление списка токенов"""
        print(f"🔄 Обновляем список токенов: {len(tokens_data)} элементов")
        
        # Очищаем существующие виджеты
        for i in reversed(range(self.tokens_layout.count())):
            child = self.tokens_layout.itemAt(i).widget()
            if child:
                child.setParent(None)
        
        # Добавляем все токены (не скрываем)
        for i, token in enumerate(tokens_data):
            print(f"  Создаем виджет для токена {i+1}: {token['symbol']}")
            token_widget = self.create_token_widget(i + 1, token)
            self.tokens_layout.addWidget(token_widget)
        
        # Добавляем растягивающийся элемент
        self.tokens_layout.addStretch()
        print(f"✅ Список токенов обновлен: {len(tokens_data)} виджетов создано")
    
    def create_token_widget(self, number, token_data):
        """Создание виджета для одного токена"""
        token_frame = ClickableFrame(token_data, self)  # Используем кастомный кликабельный frame
        token_frame.setFixedHeight(50)
        token_frame.setCursor(Qt.CursorShape.PointingHandCursor)  # Показываем что это кликабельно
        token_frame.setStyleSheet("""
            QFrame {
                background-color: #2c2c2c;
                border-radius: 6px;
                border: 1px solid #3c3c3c;
            }
            QFrame:hover {
                background-color: #3c3c3c;
                border-color: #6c5ce7;
            }
        """)
        
        # Сохраняем индекс для правильной вставки раскрывающегося окна
        token_frame.token_index = number - 1  # Индекс в layout (0-based)
        
        layout = QHBoxLayout(token_frame)
        layout.setContentsMargins(8, 5, 8, 5)
        layout.setSpacing(8)
        
        # Номер и иконка
        number_label = QLabel(str(number))
        number_label.setStyleSheet("font-size: 12px; font-weight: bold; color: #a0a0a0;")
        number_label.setFixedWidth(20)
        layout.addWidget(number_label)
        
        # Иконка токена (реальная иконка)
        icon_label = QLabel()
        icon_path = self.get_token_icon_path(token_data['symbol'])
        if icon_path and os.path.exists(icon_path):
            pixmap = QPixmap(icon_path)
            # Делаем иконку круглой
            rounded_pixmap = self.create_rounded_pixmap(pixmap, 24)
            icon_label.setPixmap(rounded_pixmap)
        else:
            # Заглушка если иконка не найдена
            icon_label.setText("●")
            icon_label.setStyleSheet(f"font-size: 16px; color: {self.get_token_color(token_data['symbol'])};")
        icon_label.setFixedWidth(24)
        icon_label.setFixedHeight(24)
        layout.addWidget(icon_label)
        
        # Информация о токене
        token_info_layout = QVBoxLayout()
        token_info_layout.setSpacing(2)
        
        # Символ токена
        symbol_label = QLabel(token_data['symbol'])
        symbol_label.setStyleSheet("font-size: 12px; font-weight: bold; color: #ffffff;")
        token_info_layout.addWidget(symbol_label)
        
        # Курс токена
        price_label = QLabel(token_data.get('price_usd', '$0.0000'))
        price_label.setStyleSheet("font-size: 10px; color: #a0a0a0;")
        token_info_layout.addWidget(price_label)
        
        # Изменение цены
        change_label = QLabel(token_data['change'])
        change_label.setStyleSheet(f"font-size: 10px; color: {token_data['change_color']};")
        token_info_layout.addWidget(change_label)
        
        layout.addLayout(token_info_layout)
        
        # Растягивающийся элемент
        layout.addStretch()
        
        # Баланс и USD значение (в контейнере с минимальной шириной)
        value_container = QWidget()
        value_container.setMinimumWidth(150)  # Минимальная ширина для видимости балансов
        value_container.setMaximumWidth(250)  # Максимальная ширина чтобы не растягивалось слишком
        value_container.setStyleSheet("background-color: transparent;")  # Прозрачный фон
        value_layout = QVBoxLayout(value_container)
        value_layout.setAlignment(Qt.AlignmentFlag.AlignRight)
        value_layout.setSpacing(1)
        value_layout.setContentsMargins(0, 0, 0, 0)
        
        # Баланс токена (жирный белый, сверху)
        balance_label = QLabel(f"{token_data['balance']} {token_data['symbol']}")
        balance_label.setStyleSheet("font-size: 12px; font-weight: bold; color: #ffffff;")
        balance_label.setAlignment(Qt.AlignmentFlag.AlignRight)
        balance_label.setWordWrap(True)  # Перенос текста если нужно
        value_layout.addWidget(balance_label)
        
        # USD значение (подтекст, меньше, снизу)
        usd_label = QLabel(token_data['usd_value'])
        usd_label.setStyleSheet("font-size: 9px; color: #888888;")
        usd_label.setAlignment(Qt.AlignmentFlag.AlignRight)
        value_layout.addWidget(usd_label)
        
        layout.addWidget(value_container)
        
        return token_frame
    
    def get_token_color(self, symbol):
        """Получение цвета для иконки токена"""
        colors = {
            "BTC": "#f7931a",
            "ETH": "#627eea", 
            "USDT": "#26a17b",
            "BNB": "#f3ba2f",
            "SOL": "#14f195",
            "XRP": "#23292f",
            "MATIC": "#8247e5"
        }
        return colors.get(symbol, "#6c5ce7")
    
    def get_token_icon_path(self, symbol):
        """Получение пути к иконке токена"""
        # Маппинг символов токенов к именам файлов иконок
        icon_mapping = {
            "BTC": "btc.png",
            "ETH": "eth.png",
            "USDT": "usdt.png",
            "USDC": "usdc.png",
            "BNB": "bnb.png",
            "SOL": "sol.png",
            "XRP": "xrp.png",
                    "MATIC": "poly.png",
                    "POL": "poly.png",
            "ADA": "ada.png",
            "DOT": "dot.png",
            "DOGE": "doge.png",
            "AVAX": "avax.png",
            "LINK": "link.png",
            "UNI": "uni.png",
            "LTC": "ltc.png",
            "BCH": "bch.png",
            "ATOM": "atom.png",
            "FIL": "fil.png",
            "TRX": "trx.png",
            "ETC": "etc.png",
            "XLM": "xlm.png",
            "VET": "vet.png",
            "ICP": "icp.png",
            "THETA": "theta.png",
            "FTM": "ftm.png",
            "AAVE": "aave.png",
            "SUSHI": "sushi.png",
            "COMP": "comp.png",
            "MKR": "mkr.png",
            "SNX": "snx.png",
            "YFI": "yfi.png",
            "CRV": "crv.png",
            "BAL": "bal.png",
            "1INCH": "1inch.png",
            "CAKE": "cake.png",
            "SAND": "sand.png",
            "MANA": "mana.png",
            "AXS": "axs.png",
            "CHZ": "chz.png",
            "ENJ": "enj.png",
            "BAT": "bat.png",
            "ZRX": "zrx.png",
            "KNC": "knc.png",
            "REN": "ren.png",
            "LRC": "lrc.png",
            "OMG": "omg.png",
            "STORJ": "storj.png",
            "REP": "rep.png",
            "KAVA": "kava.png",
            "BAND": "band.png",
            "NMR": "nmr.png",
            "CVC": "cvc.png",
            "GNT": "gnt.png",
            "DNT": "dnt.png",
            "FUN": "fun.png",
            "REQ": "req.png",
            "VIB": "vib.png",
            "MCO": "mco.png",
            "WTC": "wtc.png",
            "LEND": "lend.png",
            "BNT": "bnt.png",
            "KMD": "kmd.png",
            "LSK": "lsk.png",
            "QTUM": "qtum.png",
            "NEO": "neo.png",
            "WAVES": "waves.png",
            "STRAT": "strat.png",
            "PIVX": "pivx.png",
            "ARK": "ark.png",
            "MONA": "mona.png",
            "FCT": "fct.png",
            "MAID": "maid.png",
            "XEM": "xem.png",
            "DASH": "dash.png",
            "ZEC": "zec.png",
            "XMR": "xmr.png",
            "DCR": "dcr.png",
            "SC": "sc.png",
            "DGB": "dgb.png",
            "DGD": "dgd.png",
            "GNO": "gno.png",
            "ICN": "icn.png",
            "MLN": "mln.png",
            "SWT": "swt.png",
            "BNC": "bnc.png",
            "RLC": "rlc.png",
            "GUP": "gup.png",
            "HMQ": "hmq.png",
            "ANT": "ant.png",
            "BAT": "bat.png",
            "ZRX": "zrx.png",
            "KNC": "knc.png",
            "REN": "ren.png",
            "LRC": "lrc.png",
            "OMG": "omg.png",
            "STORJ": "storj.png",
            "REP": "rep.png",
            "KAVA": "kava.png",
            "BAND": "band.png",
            "NMR": "nmr.png",
            "CVC": "cvc.png",
            "GNT": "gnt.png",
            "DNT": "dnt.png",
            "FUN": "fun.png",
            "REQ": "req.png",
            "VIB": "vib.png",
            "MCO": "mco.png",
            "WTC": "wtc.png",
            "LEND": "lend.png",
            "BNT": "bnt.png",
            "KMD": "kmd.png",
            "LSK": "lsk.png",
            "QTUM": "qtum.png",
            "NEO": "neo.png",
            "WAVES": "waves.png",
            "STRAT": "strat.png",
            "PIVX": "pivx.png",
            "ARK": "ark.png",
            "MONA": "mona.png",
            "FCT": "fct.png",
            "MAID": "maid.png",
            "XEM": "xem.png",
            "DASH": "dash.png",
            "ZEC": "zec.png",
            "XMR": "xmr.png",
            "DCR": "dcr.png",
            "SC": "sc.png",
            "DGB": "dgb.png",
            "DGD": "dgd.png",
            "GNO": "gno.png",
            "ICN": "icn.png",
            "MLN": "mln.png",
            "SWT": "swt.png",
            "BNC": "bnc.png",
            "RLC": "rlc.png",
            "GUP": "gup.png",
            "HMQ": "hmq.png",
            "ANT": "ant.png"
        }
        
        icon_filename = icon_mapping.get(symbol.upper())
        if icon_filename:
            return f"/home/maxport/Рабочий стол/AI cursor/config/token_icons/{icon_filename}"
        return None
    
    def create_rounded_pixmap(self, pixmap, size):
        """Создание круглой иконки из обычной"""
        # Масштабируем иконку до нужного размера
        scaled_pixmap = pixmap.scaled(size, size, Qt.AspectRatioMode.KeepAspectRatio, Qt.TransformationMode.SmoothTransformation)
        
        # Создаем круглую маску
        rounded_pixmap = QPixmap(size, size)
        rounded_pixmap.fill(Qt.GlobalColor.transparent)
        
        from PyQt6.QtGui import QPainter, QBrush
        painter = QPainter(rounded_pixmap)
        painter.setRenderHint(QPainter.RenderHint.Antialiasing)
        
        # Рисуем круглую область
        painter.setBrush(QBrush(scaled_pixmap))
        painter.setPen(Qt.PenStyle.NoPen)
        painter.drawEllipse(0, 0, size, size)
        
        painter.end()
        return rounded_pixmap
    
    def get_token_prices(self, symbols):
        """Получение курсов токенов ТОЛЬКО через 0x API v2"""
        prices = {}
        
        # Проверяем кэш перед запросом
        if not hasattr(self, '_cached_prices'):
            self._cached_prices = {}
        
        cached_symbols = []
        missing_symbols = []
        
        for symbol in symbols:
            cache_key = symbol.upper()
            if cache_key in self._cached_prices:
                cached_data = self._cached_prices[cache_key]
                # Проверяем время кэша (5 минут)
                if isinstance(cached_data, dict) and 'timestamp' in cached_data:
                    if time.time() - cached_data['timestamp'] < 300:  # 5 минут
                        prices[symbol] = {
                            'price_usd': cached_data.get('price_usd', 0),
                            'change_24h': cached_data.get('change_24h', 0)
                        }
                        cached_symbols.append(symbol)
                        continue
            missing_symbols.append(symbol)
        
        # Если все цены в кэше, возвращаем сразу
        if not missing_symbols:
            print(f"✅ Все курсы из кэша: {len(cached_symbols)} токенов")
            return prices
        
        print(f"🔄 Загружаем курсы через 0x API для {len(missing_symbols)} токенов...")
        
        # Получаем все курсы через 0x API v2
        try:
            from plugins.zerox_v2_integration import ZeroXV2Integration
            
            # Получаем API ключ 0x из настроек (пробуем разные источники)
            api_key = None
            
            # Способ 1: Через api_manager
            if hasattr(self, 'main_window') and hasattr(self.main_window, 'api_manager'):
                api_keys = self.main_window.api_manager.get_api_keys()
                for key_info in api_keys:
                    if key_info.get('provider') == '0x':
                        api_key = key_info.get('key')
                        break
            
            # Способ 2: Прямо из config/RPC.txt
            if not api_key:
                try:
                    import os
                    config_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'config', 'RPC.txt')
                    if os.path.exists(config_path):
                        with open(config_path, 'r', encoding='utf-8') as f:
                            for line in f:
                                line = line.strip()
                                if line.startswith('ZEROX_API_KEY='):
                                    api_key = line.split('=', 1)[1].strip()
                                    print(f"✅ API ключ 0x загружен из config/RPC.txt")
                                    break
                except Exception as e:
                    print(f"⚠️ Ошибка чтения config/RPC.txt: {e}")
            
            if not api_key:
                print("⚠️ API ключ 0x не найден, используем без ключа")
                api_key = ""
            
            # Определяем chain_id для текущей сети
            chain_id = 137  # Polygon по умолчанию
            if self.current_network.lower() == 'ethereum':
                chain_id = 1
            elif self.current_network.lower() == 'bsc':
                chain_id = 56
            elif self.current_network.lower() == 'arbitrum':
                chain_id = 42161
            elif self.current_network.lower() == 'optimism':
                chain_id = 10
            elif self.current_network.lower() == 'avalanche':
                chain_id = 43114
            elif self.current_network.lower() == 'base':
                chain_id = 8453
            
            # Получаем адрес USDC для текущей сети (базовый токен)
            usdc_addresses = {
                137: "0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174",  # Polygon
                1: "0xA0b86991c6218b36c1d19D4a2e9Eb0cE3606eB48",      # Ethereum
                56: "0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d",      # BSC
                42161: "0xaf88d065e77c8cC2239327C5EDb3A432268e5831",   # Arbitrum
                10: "0x0b2C639c533813f4Aa9D7837CAf62653d097Ff85",      # Optimism
                43114: "0xB97EF9Ef8734C71904D8002F8b6Bc66Dd9c48a6E",   # Avalanche
                8453: "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913",    # Base
            }
            
            base_token_address = usdc_addresses.get(chain_id, usdc_addresses[137])
            
            # Настройки комиссии 0x
            wallet_address = None
            if self.wallet and hasattr(self.wallet, 'account') and self.wallet.account:
                wallet_address = self.wallet.account.address
            fee_cfg = settings_manager.get_zerox_fee_config(wallet_address) if settings_manager else {}
            fee_recipient = fee_cfg.get("recipient")
            buy_fee = fee_cfg.get("buy_token_percentage", 0.0005)
            sell_fee = fee_cfg.get("sell_token_percentage", 0.0)
            affiliate_address = fee_cfg.get("affiliate_address") or fee_recipient
            
            # Инициализируем 0x API
            zerox_api = ZeroXV2Integration(
                api_key,
                fee_recipient=fee_recipient,
                buy_token_percentage_fee=float(buy_fee),
                sell_token_percentage_fee=float(sell_fee),
                affiliate_address=affiliate_address
            )
            
            # Загружаем цены для каждого токена
            for symbol in missing_symbols:
                try:
                    # Пропускаем специальные случаи
                    if symbol.upper() in ['USDC', 'USDT']:
                        prices[symbol] = {
                            'price_usd': 1.0,
                            'change_24h': 0,
                            'timestamp': time.time()
                        }
                        continue
                    
                    # Получаем адрес токена
                    token_address = zerox_api._get_token_address(symbol, chain_id)
                    if not token_address:
                        print(f"⚠️ Адрес токена {symbol} не найден для сети {chain_id}")
                        # Пробуем fallback через CoinGecko
                        price = self._get_coingecko_price(symbol)
                        if price:
                            prices[symbol] = {
                                'price_usd': price,
                                'change_24h': 0,
                                'timestamp': time.time()
                            }
                        continue
                    
                    # Пропускаем запрос, если token_address == base_token_address
                    if token_address.lower() == base_token_address.lower():
                        prices[symbol] = {
                            'price_usd': 1.0,
                            'change_24h': 0,
                            'timestamp': time.time()
                        }
                        continue
                    
                    # Получаем decimals токена
                    sell_decimals = zerox_api._get_token_decimals(symbol, chain_id)
                    if sell_decimals is None:
                        sell_decimals = 18  # По умолчанию 18
                    
                    # Формируем параметры для запроса цены
                    # Продаем 1 токен, покупаем USDC
                    sell_amount = str(int(10 ** sell_decimals))  # 1 токен в базовых единицах
                    
                    url = f"{zerox_api.base_url}/price"
                    params = {
                        'chainId': str(chain_id),
                        'sellToken': token_address,
                        'buyToken': base_token_address,
                        'sellAmount': sell_amount,
                    }
                    
                    headers = {}
                    if api_key:
                        headers['0x-api-key'] = api_key
                    
                    response = zerox_api.session.get(url, params=params, headers=headers, timeout=10)
                    response.raise_for_status()
                    
                    data = response.json()
                    
                    # Парсим ответ v2 API
                    buy_amount_raw = data.get('buyAmount', '0')
                    
                    if buy_amount_raw == '0':
                        print(f"⚠️ Нулевая цена от 0x для {symbol}")
                        # Пробуем CoinGecko
                        price = self._get_coingecko_price(symbol)
                        if price:
                            prices[symbol] = {
                                'price_usd': price,
                                'change_24h': 0,
                                'timestamp': time.time()
                            }
                        continue
                    
                    # Конвертируем в цену (USDC имеет 6 decimals)
                    price_usd = float(buy_amount_raw) / (10 ** 6)  # USDC всегда 6 decimals
                    
                    prices[symbol] = {
                        'price_usd': price_usd,
                        'change_24h': 0,  # 0x API не предоставляет change_24h
                        'timestamp': time.time()
                    }
                    
                    print(f"✅ Получена цена {symbol}: ${price_usd:.4f}")
                    
                except Exception as e:
                    print(f"⚠️ Ошибка получения цены {symbol} через 0x: {e}")
                    # Пробуем CoinGecko как fallback
                    try:
                        price = self._get_coingecko_price(symbol)
                        if price:
                            prices[symbol] = {
                                'price_usd': price,
                                'change_24h': 0,
                                'timestamp': time.time()
                            }
                    except:
                        pass
            
            print(f"✅ Всего загружено цен: {len(prices)} из {len(missing_symbols)}")
            return prices
        
        except Exception as e:
            print(f"❌ Ошибка получения курсов через 0x API: {e}")
            import traceback
            traceback.print_exc()
            return prices
    
    def _get_coingecko_price(self, symbol: str) -> float:
        """Получает цену из CoinGecko как fallback"""
        try:
            import requests
            
            # Маппинг символов на CoinGecko IDs
            coingecko_ids = {
                'POL': 'polygon',
                'MATIC': 'polygon',
                'ETH': 'ethereum',
                'WETH': 'ethereum',
                'BTC': 'bitcoin',
                'WBTC': 'bitcoin',
                'USDC': 'usd-coin',
                'USDT': 'tether',
                'DAI': 'dai',
                'LINK': 'chainlink',
                'AAVE': 'aave',
                'UNI': 'uniswap',
            }
            
            gecko_id = coingecko_ids.get(symbol.upper())
            if not gecko_id:
                return None
            
            url = f"https://api.coingecko.com/api/v3/simple/price"
            params = {
                'ids': gecko_id,
                'vs_currencies': 'usd',
                'include_24hr_change': 'true'
            }
            
            response = requests.get(url, params=params, timeout=5)
            response.raise_for_status()
            
            data = response.json()
            if gecko_id in data:
                price = data[gecko_id].get('usd', 0)
                if price:
                    return float(price)
            
            return None
            
        except Exception as e:
            print(f"⚠️ Ошибка CoinGecko для {symbol}: {e}")
            return None
    
    def exit_wallet(self):
        """Выход из кошелька - блокируем кошелек и возвращаемся в окно приветствия"""
        print("🔒 Выход из кошелька - блокируем кошелек...")
        
        # Очищаем состояние сессии
        if self.wallet and hasattr(self.wallet, 'clear_session_state'):
            print("🔒 Очищаем состояние сессии...")
            self.wallet.clear_session_state()
            print("✅ Состояние сессии очищено")
        
        # Блокируем кошелек (очищаем account)
        if self.wallet and hasattr(self.wallet, 'account'):
            print(f"🔒 Блокируем кошелек: {self.wallet.account.address if self.wallet.account else 'None'}")
            self.wallet.account = None
            print("✅ Кошелек заблокирован")
        
        # Обновляем виджет кошелька (вернет окно приветствия)
        if self.main_window and hasattr(self.main_window, 'update_wallet_widget'):
            print("🔄 Возвращаемся в окно приветствия...")
            self.main_window.update_wallet_widget()
            print("✅ Возврат в окно приветствия выполнен")
    
    def get_pol_price_fallback(self):
        """Получение курса POL из fallback источника"""
        try:
            import requests
            url = "https://api.coingecko.com/api/v3/simple/price"
            params = {'ids': 'polygon', 'vs_currencies': 'usd'}
            response = requests.get(url, params=params, timeout=5)
            response.raise_for_status()
            data = response.json()
            if 'polygon' in data and 'usd' in data['polygon']:
                price = data['polygon']['usd']
                print(f"✅ Получен курс POL из fallback: ${price}")
                return price
        except Exception as e:
            print(f"❌ Ошибка получения курса POL: {e}")
        return 0.2  # По умолчанию возвращаем примерный курс
    
    def toggle_exchange_form(self):
        """Переключение видимости формы обмена"""
        if hasattr(self, 'exchange_form_widget'):
            visible = self.exchange_form_widget.isVisible()
            self.exchange_form_widget.setVisible(not visible)
            
            # Загружаем токены при первом открытии (ленивая загрузка)
            if not visible and hasattr(self.exchange_form_widget, '_tokens_loaded'):
                if not self.exchange_form_widget._tokens_loaded:
                    print("⏳ Загрузка токенов для обмена...")
                    self.exchange_form_widget._update_token_lists()
                    self.exchange_form_widget._tokens_loaded = True
                    print("✅ Токены загружены")
            
            # Скрываем/показываем другие элементы для освобождения места
            if hasattr(self, 'tabs_widget'):
                self.tabs_widget.setVisible(visible)  # Скрываем вкладки когда форма открыта
            if hasattr(self, 'token_scroll_area'):
                self.token_scroll_area.setVisible(visible)  # Скрываем список токенов
            if hasattr(self, 'add_token_button'):
                self.add_token_button.setVisible(visible)  # Скрываем кнопку добавления
            if hasattr(self, 'add_token_form_widget'):
                self.add_token_form_widget.setVisible(False)  # Скрываем форму добавления токена
            
            # Скрываем другие формы
            if hasattr(self, 'send_form_widget'):
                self.send_form_widget.setVisible(False)
            if hasattr(self, 'receive_form_widget'):
                self.receive_form_widget.setVisible(False)
            
            print(f"Форма обмена: {'открыта' if not visible else 'закрыта'}")
    
    def create_exchange_form(self):
        """Создание формы обмена"""
        # Используем полноценную панель обмена SwapPanel
        # Импортируем здесь, чтобы избежать циклической зависимости
        from plugins.web3_wallet import SwapPanel
        # Передаем текущую сеть из основного окна кошелька
        swap_panel = SwapPanel(parent=self, wallet=self.wallet, main_window=self.main_window, current_network=self.current_network)
        return swap_panel
    
    def confirm_exchange(self):
        """Подтверждение обмена"""
        print("Обмен подготовлен")
    
    def toggle_send_form(self):
        """Переключение видимости формы отправки"""
        if hasattr(self, 'send_form_widget'):
            visible = self.send_form_widget.isVisible()
            self.send_form_widget.setVisible(not visible)
            
            # Скрываем другие формы и список токенов
            if not visible:
                if hasattr(self, 'exchange_form_widget'):
                    self.exchange_form_widget.setVisible(False)
                if hasattr(self, 'receive_form_widget'):
                    self.receive_form_widget.setVisible(False)
                if hasattr(self, 'tabs_widget'):
                    self.tabs_widget.setVisible(False)
                if hasattr(self, 'token_scroll_area'):
                    self.token_scroll_area.setVisible(False)
                if hasattr(self, 'add_token_button'):
                    self.add_token_button.setVisible(False)
                    
                # Обновляем список токенов для отправки
                if hasattr(self, 'send_token_combo'):
                    self._update_send_token_list()
            else:
                # Показываем список токенов обратно
                if hasattr(self, 'tabs_widget'):
                    self.tabs_widget.setVisible(True)
                if hasattr(self, 'token_scroll_area'):
                    self.token_scroll_area.setVisible(True)
                if hasattr(self, 'add_token_button'):
                    self.add_token_button.setVisible(True)
            
            print(f"Форма отправки: {'открыта' if not visible else 'закрыта'}")
    
    def create_send_form(self):
        """Создание формы отправки"""
        # Создаём контейнер со скроллом
        from PyQt6.QtWidgets import QScrollArea
        
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        scroll_area.setStyleSheet("""
            QScrollArea {
                border: none;
                background-color: transparent;
            }
        """)
        
        # Внутренний контейнер для содержимого
        form = QWidget()
        form.setObjectName("sendForm")
        form.setStyleSheet("""
            QWidget#sendForm {
                background-color: #2c2c2c;
                border-radius: 8px;
            }
        """)
        
        layout = QVBoxLayout(form)
        layout.setContentsMargins(15, 15, 15, 20)
        layout.setSpacing(10)
        
        # Заголовок
        title = QLabel("Отправить токены")
        title.setStyleSheet("color: #ffffff; font-size: 14px; font-weight: bold;")
        layout.addWidget(title)
        
        # Выбор токена
        token_label = QLabel("Токен:")
        token_label.setStyleSheet("color: #cccccc; font-size: 11px;")
        layout.addWidget(token_label)
        
        self.send_token_combo = QComboBox()
        self.send_token_combo.setStyleSheet("""
            QComboBox {
                background-color: #3c3c3c;
                color: #ffffff;
                border: 1px solid #4c4c4c;
                border-radius: 6px;
                padding: 8px;
                font-size: 12px;
            }
            QComboBox:hover {
                border-color: #6c5ce7;
            }
            QComboBox::drop-down {
                border: none;
            }
        """)
        layout.addWidget(self.send_token_combo)
        
        # Адрес получателя
        address_label = QLabel("Адрес получателя:")
        address_label.setStyleSheet("color: #cccccc; font-size: 11px;")
        layout.addWidget(address_label)
        
        self.send_address_input = QLineEdit()
        self.send_address_input.setPlaceholderText("0x...")
        self.send_address_input.setStyleSheet("""
            QLineEdit {
                background-color: #3c3c3c;
                color: #ffffff;
                border: 1px solid #4c4c4c;
                border-radius: 6px;
                padding: 8px;
                font-size: 12px;
            }
            QLineEdit:focus {
                border-color: #6c5ce7;
            }
        """)
        layout.addWidget(self.send_address_input)
        
        # Сумма
        amount_label = QLabel("Сумма:")
        amount_label.setStyleSheet("color: #cccccc; font-size: 11px;")
        layout.addWidget(amount_label)
        
        self.send_amount_input = QLineEdit()
        self.send_amount_input.setPlaceholderText("0.0")
        self.send_amount_input.setStyleSheet("""
            QLineEdit {
                background-color: #3c3c3c;
                color: #ffffff;
                border: 1px solid #4c4c4c;
                border-radius: 6px;
                padding: 8px;
                font-size: 12px;
            }
            QLineEdit:focus {
                border-color: #6c5ce7;
            }
        """)
        layout.addWidget(self.send_amount_input)
        
        # Кнопка отправки
        send_confirm_button = QPushButton("Отправить")
        send_confirm_button.setFixedHeight(40)
        send_confirm_button.setStyleSheet("""
            QPushButton {
                background-color: #6c5ce7;
                color: white;
                font-weight: bold;
                font-size: 13px;
                border-radius: 8px;
            }
            QPushButton:hover {
                background-color: #5a4ac5;
            }
            QPushButton:pressed {
                background-color: #4a3aa0;
            }
        """)
        send_confirm_button.clicked.connect(self.confirm_send)
        layout.addWidget(send_confirm_button)
        
        # Добавляем растяжку в конец
        layout.addStretch()
        
        # Устанавливаем содержимое в scroll area
        scroll_area.setWidget(form)
        
        return scroll_area
    
    def _update_send_token_list(self):
        """Обновление списка токенов в форме отправки"""
        if not hasattr(self, 'send_token_combo'):
            return
            
        self.send_token_combo.clear()
        
        # Добавляем нативный токен
        native_token = self.current_network.upper() if self.current_network != "Polygon" else "POL"
        self.send_token_combo.addItem(native_token)
        
        # Добавляем токены из балансов
        if hasattr(self, 'token_balances') and self.token_balances:
            for token_symbol in self.token_balances.keys():
                if token_symbol != native_token:
                    self.send_token_combo.addItem(token_symbol)
    
    def confirm_send(self):
        """Подтверждение отправки"""
        if not hasattr(self, 'send_address_input') or not hasattr(self, 'send_amount_input') or not hasattr(self, 'send_token_combo'):
            print("❌ Форма отправки не инициализирована")
            return
            
        to_address = self.send_address_input.text().strip()
        amount_str = self.send_amount_input.text().strip()
        token = self.send_token_combo.currentText()
        
        if not to_address or not amount_str or not token:
            print("❌ Заполните все поля")
            return
            
        try:
            amount = float(amount_str)
            if amount <= 0:
                print("❌ Сумма должна быть больше 0")
                return
        except ValueError:
            print("❌ Некорректная сумма")
            return
            
        # Проверка адреса
        if not to_address.startswith('0x') or len(to_address) != 42:
            print("❌ Некорректный адрес получателя")
            return
            
        print(f"✅ Отправка {amount} {token} на адрес {to_address}")
        print("⏳ Функционал в разработке...")
        # TODO: Реализовать отправку через Web3
    
    def toggle_receive_form(self):
        """Переключение видимости формы получения"""
        if hasattr(self, 'receive_form_widget'):
            visible = self.receive_form_widget.isVisible()
            self.receive_form_widget.setVisible(not visible)
            
            # Скрываем другие формы и список токенов
            if not visible:
                if hasattr(self, 'exchange_form_widget'):
                    self.exchange_form_widget.setVisible(False)
                if hasattr(self, 'send_form_widget'):
                    self.send_form_widget.setVisible(False)
                if hasattr(self, 'tabs_widget'):
                    self.tabs_widget.setVisible(False)
                if hasattr(self, 'token_scroll_area'):
                    self.token_scroll_area.setVisible(False)
                if hasattr(self, 'add_token_button'):
                    self.add_token_button.setVisible(False)
                    
                # Обновляем адрес и QR-код в форме получения
                if self.wallet and hasattr(self.wallet, 'account') and self.wallet.account:
                    if hasattr(self, 'receive_address_label'):
                        self.receive_address_label.setText(self.wallet.account.address)
                    if hasattr(self, 'qr_code_label'):
                        self._generate_qr_code(self.wallet.account.address)
            else:
                # Показываем список токенов обратно
                if hasattr(self, 'tabs_widget'):
                    self.tabs_widget.setVisible(True)
                if hasattr(self, 'token_scroll_area'):
                    self.token_scroll_area.setVisible(True)
                if hasattr(self, 'add_token_button'):
                    self.add_token_button.setVisible(True)
            
            print(f"Форма получения: {'открыта' if not visible else 'закрыта'}")
    
    def create_receive_form(self):
        """Создание формы получения"""
        # Создаём контейнер со скроллом
        from PyQt6.QtWidgets import QScrollArea
        
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarPolicy.ScrollBarAlwaysOff)
        scroll_area.setStyleSheet("""
            QScrollArea {
                border: none;
                background-color: transparent;
            }
        """)
        
        # Внутренний контейнер для содержимого
        form = QWidget()
        form.setObjectName("receiveForm")
        form.setStyleSheet("""
            QWidget#receiveForm {
                background-color: #2c2c2c;
                border-radius: 8px;
            }
        """)
        
        layout = QVBoxLayout(form)
        layout.setContentsMargins(15, 15, 15, 20)  # Увеличен нижний отступ
        layout.setSpacing(12)
        
        # Заголовок
        title = QLabel("Получить токены")
        title.setStyleSheet("color: #ffffff; font-size: 14px; font-weight: bold;")
        layout.addWidget(title)
        
        # Информационный текст
        info_label = QLabel("Для получения токенов отправьте их на ваш адрес:")
        info_label.setWordWrap(True)
        info_label.setStyleSheet("color: #cccccc; font-size: 10px;")
        layout.addWidget(info_label)
        
        # Адрес кошелька
        address_container = QFrame()
        address_container.setStyleSheet("""
            QFrame {
                background-color: #3c3c3c;
                border: 1px solid #4c4c4c;
                border-radius: 8px;
                padding: 10px;
            }
        """)
        address_layout = QVBoxLayout(address_container)
        address_layout.setSpacing(6)
        
        address_title = QLabel("Ваш адрес:")
        address_title.setStyleSheet("color: #888888; font-size: 9px; font-weight: bold;")
        address_layout.addWidget(address_title)
        
        self.receive_address_label = QLabel("...")
        self.receive_address_label.setWordWrap(True)
        self.receive_address_label.setStyleSheet("color: #ffffff; font-size: 11px; font-weight: bold;")
        if self.wallet and hasattr(self.wallet, 'account') and self.wallet.account:
            self.receive_address_label.setText(self.wallet.account.address)
        address_layout.addWidget(self.receive_address_label)
        
        layout.addWidget(address_container)
        
        # Кнопка копирования
        copy_button = QPushButton("📋 Копировать адрес")
        copy_button.setFixedHeight(35)
        copy_button.setStyleSheet("""
            QPushButton {
                background-color: #6c5ce7;
                color: white;
                font-weight: bold;
                font-size: 12px;
                border-radius: 8px;
            }
            QPushButton:hover {
                background-color: #5a4ac5;
            }
            QPushButton:pressed {
                background-color: #4a3aa0;
            }
        """)
        copy_button.clicked.connect(self.copy_address)
        layout.addWidget(copy_button)
        
        # QR код (заглушка - генерируется только при открытии формы)
        self.qr_code_label = QLabel("Загрузка QR...")
        self.qr_code_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.qr_code_label.setFixedSize(200, 200)
        self.qr_code_label.setStyleSheet("""
            QLabel {
                background-color: #3c3c3c;
                border: 1px solid #4c4c4c;
                border-radius: 8px;
                color: #888888;
                font-size: 11px;
            }
        """)
        
        layout.addWidget(self.qr_code_label, 0, Qt.AlignmentFlag.AlignCenter)
        
        # Добавляем растяжку в конец для центрирования содержимого
        layout.addStretch()
        
        # Устанавливаем содержимое в scroll area
        scroll_area.setWidget(form)
        
        return scroll_area
    
    def _generate_qr_code(self, address: str):
        """Генерация QR-кода для адреса кошелька"""
        try:
            import qrcode
            from PyQt6.QtGui import QPixmap, QImage
            from io import BytesIO
            
            # Создаем QR-код
            qr = qrcode.QRCode(
                version=1,
                error_correction=qrcode.constants.ERROR_CORRECT_L,
                box_size=10,
                border=2,
            )
            qr.add_data(address)
            qr.make(fit=True)
            
            # Генерируем изображение
            img = qr.make_image(fill_color="black", back_color="white")
            
            # Конвертируем PIL Image в QPixmap
            buffer = BytesIO()
            img.save(buffer, format='PNG')
            buffer.seek(0)
            
            qimage = QImage()
            qimage.loadFromData(buffer.getvalue())
            pixmap = QPixmap.fromImage(qimage)
            
            # Масштабируем до размера label
            scaled_pixmap = pixmap.scaled(
                190, 190,
                Qt.AspectRatioMode.KeepAspectRatio,
                Qt.TransformationMode.SmoothTransformation
            )
            
            self.qr_code_label.setPixmap(scaled_pixmap)
            print(f"✅ QR-код успешно сгенерирован для адреса: {address}")
            
        except Exception as e:
            print(f"❌ Ошибка генерации QR-кода: {e}")
            self.qr_code_label.setText("Ошибка\nгенерации\nQR-кода")
            self.qr_code_label.setStyleSheet("""
                QLabel {
                    background-color: #3c3c3c;
                    border: 1px solid #4c4c4c;
                    border-radius: 8px;
                    color: #ff6b6b;
                    font-size: 10px;
                }
            """)
    
    def copy_address(self):
        """Копирование адреса"""
        if self.wallet and hasattr(self.wallet, 'account') and self.wallet.account:
            from PyQt6.QtWidgets import QApplication
            clipboard = QApplication.clipboard()
            clipboard.setText(self.wallet.account.address)
            print(f"✅ Адрес скопирован: {self.wallet.account.address}")
    
    def toggle_add_token_form(self):
        """Переключение видимости формы добавления токена"""
        if hasattr(self, 'add_token_form_widget'):
            visible = self.add_token_form_widget.isVisible()
            self.add_token_form_widget.setVisible(not visible)
            print(f"Форма добавления токена: {'открыта' if not visible else 'закрыта'}")
    
    def create_add_token_form(self):
        """Создание формы добавления токена"""
        form_container = QFrame()
        form_container.setStyleSheet("""
            QFrame {
                background-color: #2c2c2c;
                border: 1px solid #6c5ce7;
                border-radius: 6px;
            }
        """)
        form_container.setMinimumHeight(200)
        
        layout = QVBoxLayout(form_container)
        layout.setSpacing(12)
        layout.setContentsMargins(16, 12, 16, 12)
        
        # Заголовок
        title = QLabel("➕ Добавить токен")
        title.setStyleSheet("font-size: 12px; font-weight: bold; color: #ffffff;")
        layout.addWidget(title)
        
        # Поле для адреса контракта
        addr_label = QLabel("Адрес контракта токена:")
        addr_label.setStyleSheet("font-size: 10px; color: #a0a0a0;")
        layout.addWidget(addr_label)
        
        addr_input = QLineEdit()
        addr_input.setPlaceholderText("Вставьте адрес контракта ERC20...")
        addr_input.setStyleSheet("""
            QLineEdit {
                background-color: #1a1a1a;
                border: 1px solid #3c3c3c;
                border-radius: 4px;
                padding: 8px;
                color: #ffffff;
                font-size: 10px;
            }
            QLineEdit:focus {
                border: 1px solid #6c5ce7;
            }
        """)
        layout.addWidget(addr_input)
        
        # Кнопки
        buttons_layout = QHBoxLayout()
        buttons_layout.setSpacing(8)
        
        # Кнопка добавить
        add_btn = QPushButton("✅ Добавить")
        add_btn.setFixedHeight(32)
        add_btn.setStyleSheet("""
            QPushButton {
                background-color: #27ae60;
                color: #ffffff;
                border: none;
                border-radius: 4px;
                padding: 6px 12px;
                font-size: 11px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #229954;
            }
            QPushButton:pressed {
                background-color: #1e8449;
            }
        """)
        buttons_layout.addWidget(add_btn)
        
        # Кнопка отмена
        cancel_btn = QPushButton("❌ Отмена")
        cancel_btn.setFixedHeight(32)
        cancel_btn.setStyleSheet("""
            QPushButton {
                background-color: #e74c3c;
                color: #ffffff;
                border: none;
                border-radius: 4px;
                padding: 6px 12px;
                font-size: 11px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #c0392b;
            }
            QPushButton:pressed {
                background-color: #a93226;
            }
        """)
        cancel_btn.clicked.connect(lambda: form_container.setVisible(False))
        buttons_layout.addWidget(cancel_btn)
        
        layout.addLayout(buttons_layout)
        layout.addStretch()
        
        return form_container
    
    def confirm_add_token(self):
        """Подтверждение добавления токена"""
        print("Токен добавлен")
    
    def add_token(self):
        """Добавление токена"""
        print("Добавление токена...")
    
    def show_tokens(self):
        """Показать токены"""
        print("Показываем токены...")
        self.current_view = 'tokens'
        self.update_nav_buttons()  # Обновляем подсветку кнопок
        
        # Очищаем контейнер
        print(f"💰 Очищаю tokens_layout (элементов: {self.tokens_layout.count()})")
        while self.tokens_layout.count():
            child = self.tokens_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        print("💰 Очистка завершена")
        
        # Загружаем данные токенов ТОЛЬКО если нет кэша
        import time
        if not self.tokens_data:
            print("💰 Первая загрузка токенов...")
            self.load_tokens_data()
        elif (time.time() - self.last_load_time > 30):
            print("💰 Обновляю токены (прошло >30 сек)...")
            self.load_tokens_data()
        else:
            print(f"💰 Кэш свежий, пропускаю загрузку ({len(self.tokens_data)} токенов)")
    
    def show_nft(self):
        """Показать NFT"""
        print("Показываем NFT...")
        self.current_view = 'nft'
        self.update_nav_buttons()  # Обновляем подсветку кнопок
        
        # Очищаем контейнер
        print(f"🎨 Очищаю tokens_layout (элементов: {self.tokens_layout.count()})")
        while self.tokens_layout.count():
            child = self.tokens_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        print("🎨 Очистка завершена")
        
        # Загружаем NFT
        self.load_nfts_data()
    
    def collect_tokens(self):
        """Обмен токенов"""
        print("Обмен токенов...")
        self.current_view = 'exchange'
        self.toggle_exchange_form()
    
    def send_tokens(self):
        """Отправка токенов"""
        print("Отправка токенов...")
        self.current_view = 'send'
        self.toggle_send_form()
    
    def receive_tokens(self):
        """Получение токенов"""
        print("Получение токенов...")
        self.current_view = 'receive'
        self.toggle_receive_form()
    
    def show_activity(self):
        """Показать активность (историю транзакций)"""
        import sys
        print("📊 ВЫЗВАНА ФУНКЦИЯ show_activity!", flush=True)
        sys.stdout.flush()
        print("📊 Показываем активность...", flush=True)
        sys.stdout.flush()
        
        try:
            # Переключаемся на режим активности
            print("📊 Этап 1: Переключаемся на режим активности", flush=True)
            self.current_view = 'activity'
            print("📊 current_view = 'activity'", flush=True)
            sys.stdout.flush()
            
            print("📊 Этап 2: Обновляем nav_buttons", flush=True)
            self.update_nav_buttons()  # Обновляем подсветку кнопок
            print("📊 nav_buttons обновлены", flush=True)
            sys.stdout.flush()
            
            # Очищаем контейнер токенов
            print(f"📊 Этап 3: Очищаю tokens_layout (элементов: {self.tokens_layout.count()})", flush=True)
            while self.tokens_layout.count():
                child = self.tokens_layout.takeAt(0)
                if child.widget():
                    child.widget().deleteLater()
            print("📊 tokens_layout очищен", flush=True)
            sys.stdout.flush()
            
            # Проверяем кошелек
            print("📊 Этап 4: Проверяем кошелек", flush=True)
            if not self.wallet or not hasattr(self.wallet, 'account') or not self.wallet.account:
                print("❌ Кошелек не инициализирован", flush=True)
                error_label = QLabel("❌ Кошелек не инициализирован")
                error_label.setStyleSheet("color: #e74c3c; font-size: 14px; padding: 20px;")
                error_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
                self.tokens_layout.addWidget(error_label)
                return
            
            print("✅ Кошелек инициализирован", flush=True)
            
            # Получаем адрес
            address = self.wallet.account.address
            print(f"📊 Адрес кошелька: {address}", flush=True)
            sys.stdout.flush()
            
            # Определяем сеть
            network_name = self.current_network
            print(f"📊 Сеть: {network_name}", flush=True)
            sys.stdout.flush()
            
            # === КОМПАКТНЫЙ ЗАГОЛОВОК И ФИЛЬТРЫ ===
            print("📊 Этап 5: Создаю компактный header с фильтрами", flush=True)
            
            # Главный контейнер для header + транзакций
            header_frame = QFrame()
            header_frame.setStyleSheet("QFrame { background-color: transparent; border: none; }")
            header_layout = QVBoxLayout(header_frame)
            header_layout.setContentsMargins(0, 0, 0, 0)
            header_layout.setSpacing(6)
            
            # Кнопки фильтра в компактном виде
            filter_frame = QFrame()
            filter_frame.setStyleSheet("QFrame { background-color: transparent; border: none; }")
            filter_layout = QHBoxLayout(filter_frame)
            filter_layout.setContentsMargins(0, 0, 0, 0)
            filter_layout.setSpacing(4)
            
            all_btn = QPushButton("Все")
            sent_btn = QPushButton("Отправлено")
            received_btn = QPushButton("Получено")
            swap_btn = QPushButton("Обмен")
            
            # Компактный стиль кнопок
            btn_style = "QPushButton { background-color: #2c2c2c; color: #888888; border: 1px solid #3c3c3c; border-radius: 4px; padding: 6px 10px; font-size: 10px; font-weight: bold; } QPushButton:hover { background-color: #333333; color: #ffffff; border-color: #6c5ce7; }"
            btn_active_style = "QPushButton { background-color: #6c5ce7; color: #ffffff; border: 1px solid #6c5ce7; border-radius: 4px; padding: 6px 10px; font-size: 10px; font-weight: bold; }"
            
            for btn in [all_btn, sent_btn, received_btn, swap_btn]:
                btn.setStyleSheet(btn_style)
                btn.setMaximumWidth(95)
                btn.setMinimumHeight(28)
            
            # По умолчанию "Все" активна
            self.active_filter_btn = all_btn
            all_btn.setStyleSheet(btn_active_style)
            
            filter_layout.addWidget(all_btn, 0)
            filter_layout.addWidget(sent_btn, 0)
            filter_layout.addWidget(received_btn, 0)
            filter_layout.addWidget(swap_btn, 0)
            filter_layout.addStretch()
            
            header_layout.addWidget(filter_frame)
            self.tokens_layout.addWidget(header_frame)
            
            # Сохраняем кнопки для переключения активности
            self.filter_buttons = {
                'all': all_btn,
                'sent': sent_btn,
                'received': received_btn,
                'swap': swap_btn
            }
            
            print("📊 Компактный header создан", flush=True)
            sys.stdout.flush()
            
            # === КОНТЕЙНЕР ДЛЯ ТРАНЗАКЦИЙ ===
            print("📊 Этап 7: Создаю контейнер для транзакций", flush=True)
            self.activity_tx_container = QWidget()
            self.activity_tx_container_layout = QVBoxLayout(self.activity_tx_container)
            self.activity_tx_container_layout.setContentsMargins(0, 0, 0, 0)
            self.activity_tx_container_layout.setSpacing(3)  # Минимальное расстояние
            self.activity_tx_container.setMinimumHeight(400)
            print("📊 Контейнер для транзакций создан", flush=True)
            sys.stdout.flush()
            
            # === SCROLL AREA ===
            print("📊 Этап 8: Создаю scroll area", flush=True)
            self.activity_scroll = QScrollArea()
            self.activity_scroll.setWidget(self.activity_tx_container)
            self.activity_scroll.setWidgetResizable(True)
            self.activity_scroll.setStyleSheet("QScrollArea { border: none; background-color: transparent; }")
            self.tokens_layout.addWidget(self.activity_scroll)
            print("📊 Scroll area создан и добавлен", flush=True)
            sys.stdout.flush()
            
            # === ПОКАЗЫВАЕМ ЗАГРУЗКУ ===
            print("📊 Этап 9: Показываю загрузку", flush=True)
            self.loading_label = QLabel("⏳ Загрузка истории транзакций...")
            self.loading_label.setStyleSheet("color: #6c5ce7; font-size: 14px; padding: 20px;")
            self.loading_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            self.activity_tx_container_layout.addWidget(self.loading_label)
            print("📊 Индикатор загрузки показан", flush=True)
            sys.stdout.flush()
            
            # === ПОДКЛЮЧАЕМ КНОПКИ ФИЛЬТРА ===
            print("📊 Этап 10: Подключаю кнопки фильтра", flush=True)
            all_btn.clicked.connect(lambda: self._apply_activity_filter('all'))
            sent_btn.clicked.connect(lambda: self._apply_activity_filter('sent'))
            received_btn.clicked.connect(lambda: self._apply_activity_filter('received'))
            swap_btn.clicked.connect(lambda: self._apply_activity_filter('swap'))
            print("📊 Кнопки фильтра подключены", flush=True)
            sys.stdout.flush()
            
            # === ЗАГРУЖАЕМ ИСТОРИЮ В ПОТОКЕ ===
            print(f"🧵 Этап 11: Создаю TransactionLoadThread...", flush=True)
            print(f"🧵 Адрес: {address}, Сеть: {network_name}", flush=True)
            sys.stdout.flush()
            
            self.load_thread = TransactionLoadThread(self.tx_scanner, address, network_name)
            print(f"🧵 Поток создан: {self.load_thread}", flush=True)
            
            self.load_thread.finished.connect(self._on_activity_history_loaded)
            self.load_thread.error.connect(self._on_activity_history_error)
            self.load_thread.start()
            print(f"🧵 Поток запущен", flush=True)
            sys.stdout.flush()
            
        except Exception as e:
            print(f"❌ ОШИБКА в show_activity: {e}", flush=True)
            import traceback
            print(f"📋 Traceback:\n{traceback.format_exc()}", flush=True)
            sys.stdout.flush()
    
    def _apply_activity_filter(self, filter_type):
        """Применяет фильтр к истории транзакций"""
        import sys
        
        # 🆕 Обновляем активную кнопку фильтра визуально
        btn_active_style = "QPushButton { background-color: #6c5ce7; color: #ffffff; border: 1px solid #6c5ce7; border-radius: 4px; padding: 6px 10px; font-size: 10px; font-weight: bold; }"
        btn_style = "QPushButton { background-color: #2c2c2c; color: #888888; border: 1px solid #3c3c3c; border-radius: 4px; padding: 6px 10px; font-size: 10px; font-weight: bold; } QPushButton:hover { background-color: #333333; color: #ffffff; border-color: #6c5ce7; }"
        
        try:
            if hasattr(self, 'filter_buttons') and self.filter_buttons:
                # Сбрасываем стиль всех кнопок
                for btn in list(self.filter_buttons.values()):
                    try:
                        btn.setStyleSheet(btn_style)
                    except:
                        pass
                # Активируем текущую кнопку
                if filter_type in self.filter_buttons:
                    try:
                        self.filter_buttons[filter_type].setStyleSheet(btn_active_style)
                    except:
                        pass
        except:
            pass
        
        history = getattr(self, 'full_transaction_history', [])
        print(f"🔄 Применяю фильтр '{filter_type}' к {len(history)} транзакциям", flush=True)
        sys.stdout.flush()
        
        # Очищаем контейнер
        print(f"🔄 Очищаю контейнер (элементов: {self.activity_tx_container_layout.count()})", flush=True)
        while self.activity_tx_container_layout.count():
            child = self.activity_tx_container_layout.takeAt(0)
            if child.widget():
                child.widget().deleteLater()
        print(f"🔄 Контейнер очищен", flush=True)
        sys.stdout.flush()
        
        if not history:
            print(f"⚠️ История пуста!", flush=True)
            empty_label = QLabel("Нет транзакций")
            empty_label.setStyleSheet("color: #a0a0a0; font-size: 12px; padding: 20px;")
            empty_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            self.activity_tx_container_layout.addWidget(empty_label)
            return
        
        # Группируем по датам и отображаем
        from datetime import datetime
        txs_by_date = {}
        filtered_count = 0
        
        print(f"🔄 Начинаю фильтрацию {len(history[:50])} транзакций...", flush=True)
        for tx in history[:50]:
            tx_type = tx.get('type', 'receive')  # 'send' или 'receive' из транзакции
            tx_kind = tx.get('tx_type', 'native')  # 'native' или 'token'
            
            # Определяем реальный тип операции
            # ВАЖНО: tx_kind='token' означает транзакция с токеном, но тип (send/receive) определен правильно!
            # Swap - это когда идет обмен между разными контрактами (оба send и receive от смартконтрактов)
            # Мы покажем как swap только если это явно обмен или специальная транзакция
            real_type = tx_type  # Просто используем уже определенный тип 'send' или 'receive'
            
            # Логика фильтрации
            include = False
            if filter_type == 'all':
                include = True
            elif filter_type == 'sent' and real_type == 'send':
                include = True
            elif filter_type == 'received' and real_type == 'receive':
                include = True
            elif filter_type == 'swap' and tx_kind == 'token':  # Swap = любые транзакции с токенами
                include = True
            if not include:
                continue
            
            filtered_count += 1
            
            try:
                timestamp = int(tx.get('timestamp', 0))
                tx_date = datetime.fromtimestamp(timestamp).date()
                date_str = tx_date.strftime("%d.%m.%Y")
                
                if date_str not in txs_by_date:
                    txs_by_date[date_str] = []
                txs_by_date[date_str].append(tx)
            except:
                pass
        
        print(f"🔄 Фильтровано: {filtered_count} транзакций в {len(txs_by_date)} датах", flush=True)
        sys.stdout.flush()
        
        for date_str in sorted(txs_by_date.keys(), reverse=True):
            print(f"🔄 Отображаю дату: {date_str} ({len(txs_by_date[date_str])} транзакций)", flush=True)
            date_header = QLabel(date_str)
            date_header.setStyleSheet("font-size: 11px; color: #6c5ce7; font-weight: bold; padding: 8px 0px 5px 0px;")
            self.activity_tx_container_layout.addWidget(date_header)
            
            for tx in txs_by_date[date_str]:
                try:
                    tx_kind = tx.get('tx_type', 'native')
                    token_symbol = tx.get('token_symbol') or 'POL'
                    value = float(tx.get('value', 0))
                    to_addr = tx.get('to', '')
                    to_addr_short = (to_addr[0:6] + '...' + to_addr[-4:]) if to_addr else '[контракт]'
                    status = tx.get('status', 'success')
                    timestamp = int(tx.get('timestamp', 0))
                    
                    # КАРТОЧКА - КОМПАКТНАЯ ОДНОСТРОЧНАЯ С КЛИКОМ
                    tx_frame = QFrame()
                    tx_frame.setStyleSheet("""
                        QFrame {
                            background-color: #2c2c2c;
                            border: 1px solid #3c3c3c;
                            border-radius: 6px;
                            padding: 4px 6px;
                            margin: 0px 0px;
                        }
                        QFrame:hover {
                            background-color: #333333;
                            border-color: #6c5ce7;
                        }
                    """)
                    tx_frame.setCursor(Qt.CursorShape.PointingHandCursor)
                    
                    # Горизонтальный layout - ВСЕ В ОДНУ СТРОКУ
                    main_layout = QHBoxLayout(tx_frame)
                    main_layout.setContentsMargins(0, 2, 0, 2)  # Минимальные отступы
                    main_layout.setSpacing(10)
                    
                    # === ЭЛЕМЕНТ 1: ИКОНКА ===
                    if tx_kind == 'token':
                        tx_icon = "🔄"
                    elif tx.get('type') == 'send':
                        tx_icon = "📤"
                    else:
                        tx_icon = "📥"
                    
                    icon_label = QLabel(tx_icon)
                    icon_label.setStyleSheet("""
                        QLabel {
                            font-size: 18px;
                            background-color: transparent;
                            margin: 0px;
                            padding: 0px;
                        }
                    """)
                    icon_label.setFixedSize(24, 24)
                    icon_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
                    main_layout.addWidget(icon_label, 0, Qt.AlignmentFlag.AlignVCenter)
                    
                    # === ЭЛЕМЕНТ 2: АДРЕС ===
                    addr_label = QLabel(to_addr_short)
                    addr_label.setStyleSheet("""
                        QLabel {
                            font-size: 10px;
                            color: #888888;
                            background-color: transparent;
                            margin: 0px;
                            padding: 0px;
                        }
                    """)
                    addr_label.setMinimumWidth(70)  # Адаптивная ширина
                    addr_label.setWordWrap(True)  # Обтекание текста
                    addr_label.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignVCenter)
                    main_layout.addWidget(addr_label, 0)
                    
                    # === ЭЛЕМЕНТ 3: КОЛИЧЕСТВО И ТОКЕН ===
                    if tx_kind == 'token':
                        amount_text = f"{value:.4f} {token_symbol}"
                    else:
                        amount_text = f"{value:.6f} {token_symbol}"
                    
                    # Проверяем, это ли swap транзакция с полной информацией
                    if tx.get('tx_type') == 'token_swap':
                        # Это сохраненный swap с полной информацией
                        from_token = tx.get('from_token', 'UNKNOWN')
                        to_token = tx.get('to_token', 'UNKNOWN')
                        from_amount = tx.get('from_amount', 0)
                        to_amount = tx.get('to_amount', 0)
                        amount_text = f"{from_amount:.4f} {from_token} → {to_amount:.4f} {to_token}"
                    
                    amount_label = QLabel(amount_text)
                    amount_label.setStyleSheet("""
                        QLabel {
                            font-size: 11px;
                            font-weight: bold;
                            color: #ffffff;
                            background-color: transparent;
                            margin: 0px;
                            padding: 0px;
                        }
                    """)
                    amount_label.setMinimumWidth(100)  # Адаптивная ширина
                    amount_label.setWordWrap(True)  # Обтекание текста
                    amount_label.setAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignVCenter)
                    main_layout.addWidget(amount_label, 0)
                    
                    # === ЭЛЕМЕНТ 4: РАСТЯГИВАЮЩИЙСЯ ЭЛЕМЕНТ ===
                    main_layout.addStretch()
                    
                    # === ЭЛЕМЕНТ 5: ВРЕМЯ ===
                    try:
                        time_str = datetime.fromtimestamp(timestamp).strftime("%H:%M")
                        time_label = QLabel(time_str)
                        time_label.setStyleSheet("""
                            QLabel {
                                font-size: 10px;
                                color: #666666;
                                background-color: transparent;
                                margin: 0px;
                                padding: 0px;
                            }
                        """)
                        time_label.setMinimumWidth(35)  # Адаптивная ширина
                        time_label.setAlignment(Qt.AlignmentFlag.AlignRight | Qt.AlignmentFlag.AlignVCenter)
                        main_layout.addWidget(time_label, 0, Qt.AlignmentFlag.AlignVCenter)
                    except:
                        pass
                    
                    # === ЭЛЕМЕНТ 6: СТАТУС ===
                    status_icon = "✅" if status == 'success' else "❌"
                    status_label = QLabel(status_icon)
                    status_label.setStyleSheet("""
                        QLabel {
                            font-size: 12px;
                            background-color: transparent;
                            margin: 0px;
                            padding: 0px;
                        }
                    """)
                    status_label.setFixedWidth(18)
                    status_label.setAlignment(Qt.AlignmentFlag.AlignCenter | Qt.AlignmentFlag.AlignVCenter)
                    main_layout.addWidget(status_label, 0)
                    
                    # === КЛИК ДЛЯ ОТКРЫТИЯ В БЛОКЧЕЙНСКАНЕ ===
                    tx_hash = tx.get('hash', '')
                    
                    def make_click_handler(hash_val, network_name):
                        def on_tx_click(event):
                            if hash_val:
                                import webbrowser
                                explorers = {
                                    'Polygon': f'https://polygonscan.com/tx/{hash_val}',
                                    'Ethereum': f'https://etherscan.io/tx/{hash_val}',
                                    'BSC': f'https://bscscan.com/tx/{hash_val}',
                                    'Arbitrum': f'https://arbiscan.io/tx/{hash_val}',
                                    'Optimism': f'https://optimistic.etherscan.io/tx/{hash_val}',
                                    'Avalanche': f'https://snowtrace.io/tx/{hash_val}',
                                    'Base': f'https://basescan.org/tx/{hash_val}',
                                }
                                url = explorers.get(network_name, f'https://polygonscan.com/tx/{hash_val}')
                                print(f"🔗 Открываю транзакцию в блокчейнскане: {url}")
                                webbrowser.open(url)
                        return on_tx_click
                    
                    tx_frame.mousePressEvent = make_click_handler(tx_hash, self.current_network)
                    
                    self.activity_tx_container_layout.addWidget(tx_frame)
                    
                except Exception as e:
                    print(f"⚠️ Ошибка при добавлении карточки: {e}", flush=True)
        
        print(f"✅ Фильтр '{filter_type}' показано {filtered_count} транзакций", flush=True)
        sys.stdout.flush()
        
        try:
            # Добавляем stretch в конец чтобы транзакции были в верху, а не растянуты
            print(f"🔄 Добавляю stretch и обновляю layout", flush=True)
            if hasattr(self, 'activity_tx_container_layout') and self.activity_tx_container_layout:
                self.activity_tx_container_layout.addStretch()
                print(f"🔄 Stretch добавлен", flush=True)
            else:
                print(f"⚠️ activity_tx_container_layout не найден", flush=True)
            
            # Устанавливаем минимальную высоту в зависимости от количества транзакций
            estimated_height = max(filtered_count * 35 + 50, 400)
            if hasattr(self, 'activity_tx_container') and self.activity_tx_container:
                self.activity_tx_container.setMinimumHeight(estimated_height)
                print(f"🔄 Установлена высота контейнера: {estimated_height}px", flush=True)
            
            # Обновляем layout
            if hasattr(self, 'activity_tx_container'):
                self.activity_tx_container.update()
                print(f"🔄 activity_tx_container обновлен", flush=True)
            if hasattr(self, 'activity_scroll'):
                self.activity_scroll.update()
                print(f"🔄 activity_scroll обновлен", flush=True)
            
            print(f"✅ Layout обновлен", flush=True)
            sys.stdout.flush()
        except Exception as e:
            print(f"❌ Ошибка при обновлении layout: {e}", flush=True)
            import traceback
            print(f"📋 {traceback.format_exc()}", flush=True)
            sys.stdout.flush()
    
    def _on_activity_history_loaded(self, history):
        """Обработчик загрузки истории транзакций"""
        import sys
        print(f"✅ _on_activity_history_loaded вызвана", flush=True)
        sys.stdout.flush()
        print(f"✅ Получено {len(history)} транзакций из потока", flush=True)
        sys.stdout.flush()
        
        self.full_transaction_history = history
        print(f"✅ История сохранена в self.full_transaction_history", flush=True)
        sys.stdout.flush()
        
        try:
            print(f"✅ Пытаюсь удалить loading_label", flush=True)
            sys.stdout.flush()
            self.activity_tx_container_layout.removeWidget(self.loading_label)
            self.loading_label.deleteLater()
            print(f"✅ Loading label удален", flush=True)
            sys.stdout.flush()
        except Exception as e:
            print(f"⚠️ Ошибка удаления loading label: {e}", flush=True)
            sys.stdout.flush()
        
        if not history:
            print(f"⚠️ История пуста!", flush=True)
            sys.stdout.flush()
            empty_label = QLabel("📭 История пуста")
            empty_label.setStyleSheet("color: #a0a0a0; font-size: 14px; padding: 20px;")
            empty_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            self.activity_tx_container_layout.addWidget(empty_label)
            print("⚠️ История транзакций пуста", flush=True)
            return
        
        print(f"✅ Применяю фильтр all", flush=True)
        sys.stdout.flush()
        self._apply_activity_filter('all')
        print(f"✅ Транзакции отображены", flush=True)
        sys.stdout.flush()
    
    def _on_activity_history_error(self, error):
        """Обработчик ошибки загрузки истории"""
        import sys
        print(f"❌ _on_activity_history_error вызвана", flush=True)
        print(f"❌ Ошибка загрузки: {error}", flush=True)
        sys.stdout.flush()
        
        try:
            self.activity_tx_container_layout.removeWidget(self.loading_label)
            self.loading_label.deleteLater()
        except:
            pass
        
        error_label = QLabel(f"❌ Ошибка: {error}")
        error_label.setStyleSheet("color: #e74c3c; font-size: 14px; padding: 20px;")
        error_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.activity_tx_container_layout.addWidget(error_label)
        print(f"❌ Сообщение об ошибке добавлено", flush=True)
        sys.stdout.flush()
    
    def show_token_details(self, token_data, token_index=None):
        """Показывает раскрывающееся компактное окно под токеном (toggle)"""
        # Если уже есть открытое окно
        if hasattr(self, '_current_detail_widget') and self._current_detail_widget:
            # Если кликнули на ТОМ ЖЕ токене - закрываем окно (toggle)
            if hasattr(self, '_current_detail_index') and self._current_detail_index == token_index:
                self._current_detail_widget.setParent(None)
                self._current_detail_widget.deleteLater()
                self._current_detail_widget = None
                self._current_detail_index = -1
                return
            
            # Если кликнули на ДРУГОМ токене - закрываем старое окно и открываем новое
            self._current_detail_widget.setParent(None)
            self._current_detail_widget.deleteLater()
        
        # Получаем адрес контракта
        contract_address = token_data.get('address', '')
        symbol = token_data.get('symbol', 'Unknown')
        
        # Создаем компактное раскрывающееся окно
        detail_frame = QFrame()
        detail_frame.setStyleSheet("""
            QFrame {
                background-color: #2c2c2c;
                border: 1px solid #6c5ce7;
                border-radius: 6px;
                margin: 5px 0px;
            }
        """)
        # Не устанавливаем фиксированную высоту - пусть расширяется по содержимому
        detail_frame.setMinimumHeight(80)  # Минимальная высота для кнопок
        
        layout = QVBoxLayout(detail_frame)
        layout.setSpacing(8)
        layout.setContentsMargins(12, 10, 12, 10)
        
        # Заголовок с символом
        title = QLabel(f"📋 Адрес контракта: {symbol}")
        title.setStyleSheet("font-size: 11px; font-weight: bold; color: #ffffff;")
        layout.addWidget(title)
        
        # Адрес - адаптивный контейнер с автоматическим расширением (для всех токенов)
        addr_container = QFrame()
        addr_container.setStyleSheet("""
            QFrame {
                background-color: #1a1a1a;
                border: 1px solid #3c3c3c;
                border-radius: 4px;
            }
        """)
        addr_layout = QVBoxLayout(addr_container)
        addr_layout.setContentsMargins(6, 6, 6, 6)
        addr_layout.setSpacing(0)
        
        # Показываем адрес или сообщение для нативного токена
        if contract_address and contract_address != "":
            display_text = contract_address
            is_native = False
        else:
            display_text = f"🔗 {symbol} - Нативный токен сети"
            is_native = True
        
        addr_display = QLabel(display_text)
        addr_display.setStyleSheet("""
            font-family: monospace;
            font-size: 10px;
            color: #a0a0a0;
        """)
        addr_display.setWordWrap(True)
        addr_display.setMinimumHeight(30)  # Минимальная высота для адреса
        addr_layout.addWidget(addr_display)
        
        layout.addWidget(addr_container)
        
        # Кнопки в одну линию
        button_layout = QHBoxLayout()
        button_layout.setSpacing(8)
        
        # Копировать адрес (только для ERC20 токенов, не для нативных)
        if not is_native:
            copy_button = QPushButton("📋 Копировать")
            copy_button.setFixedHeight(28)
            copy_button.setStyleSheet("""
                QPushButton {
                    background-color: #6c5ce7;
                    color: #ffffff;
                    border: none;
                    border-radius: 3px;
                    padding: 4px 8px;
                    font-size: 10px;
                    font-weight: bold;
                }
                QPushButton:hover {
                    background-color: #7d6ce8;
                }
            """)
            copy_button.clicked.connect(lambda: self.copy_to_clipboard(contract_address))
            button_layout.addWidget(copy_button)
            
            # Открыть в блокчейне
            explorer_button = QPushButton("🔗 Блокчейнскан")
            explorer_button.setFixedHeight(28)
            explorer_button.setStyleSheet("""
                QPushButton {
                    background-color: #27ae60;
                    color: #ffffff;
                    border: none;
                    border-radius: 3px;
                    padding: 4px 8px;
                    font-size: 10px;
                    font-weight: bold;
                }
                QPushButton:hover {
                    background-color: #229954;
                }
            """)
            explorer_button.clicked.connect(lambda: self.open_in_explorer(contract_address))
            button_layout.addWidget(explorer_button)
        
            # Кнопка закрытия показывается только если есть действия с адресом,
            # чтобы нативные токены не выглядели так, будто их можно удалить.
        close_button = QPushButton("✕")
        close_button.setFixedSize(28, 28)
        close_button.setStyleSheet("""
            QPushButton {
                background-color: #e74c3c;
                color: #ffffff;
                border: none;
                border-radius: 3px;
                padding: 0px;
                font-size: 12px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #c0392b;
            }
        """)
        close_button.clicked.connect(lambda: self.hide_token_details())
        button_layout.addWidget(close_button)
        
        layout.addLayout(button_layout)
        
        # Вставляем окно сразу ПОСЛЕ токена (а не в конец)
        if token_index is not None:
            # Вставляем сразу после текущего токена: token_index (токен) + 1 (вставляем после)
            insert_pos = token_index + 1
            self.tokens_layout.insertWidget(insert_pos, detail_frame)
        else:
            # Fallback - вставляем в конец перед stretch
            self.tokens_layout.insertWidget(self.tokens_layout.count() - 1, detail_frame)
        
        # Сохраняем открытое окно и его индекс для toggle
        self._current_detail_widget = detail_frame
        self._current_detail_index = token_index
    
    def hide_token_details(self):
        """Скрывает раскрывающееся окно"""
        if hasattr(self, '_current_detail_widget') and self._current_detail_widget:
            self._current_detail_widget.setParent(None)
            self._current_detail_widget.deleteLater()
            self._current_detail_widget = None
        self._current_detail_index = -1
    
    def copy_to_clipboard(self, text):
        """Копирует текст в буфер обмена"""
        from PyQt6.QtWidgets import QApplication
        clipboard = QApplication.clipboard()
        clipboard.setText(text)
        # Адрес скопирован, уведомление убрано
    
    def open_in_explorer(self, address):
        """Открывает адрес в блокчейн сканере"""
        import webbrowser
        
        # Определяем сеть и URL
        explorer_urls = {
            'ethereum': f"https://etherscan.io/address/{address}",
            'polygon': f"https://polygonscan.com/address/{address}",
            'bsc': f"https://bscscan.com/address/{address}",
            'arbitrum': f"https://arbiscan.io/address/{address}",
            'optimism': f"https://optimistic.etherscan.io/address/{address}",
            'avalanche': f"https://snowtrace.io/address/{address}",
            'base': f"https://basescan.org/address/{address}",
            'linea': f"https://lineascan.build/address/{address}"
        }
        
        # Получаем текущую сеть из balance_manager
        current_network = getattr(self.balance_manager, 'current_network', 'polygon')
        url = explorer_urls.get(current_network, explorer_urls['polygon'])
        
        print(f"🔗 Открываю {url}")
        webbrowser.open(url)