import requests
import json
import time
from typing import Dict, List, Optional, Tuple, Any
from decimal import Decimal
from web3 import Web3
from plugins.token_utils import PREDEFINED_TOKENS

class DEXIntegration:
    """
    Интеграция с DEX агрегаторами для получения лучших курсов обмена.
    Поддерживает: 1inch, Uniswap, PancakeSwap, 0x Protocol
    """
    
    def __init__(self):
        self.session = requests.Session()
        self.session.headers.update({
            'User-Agent': 'MaxportAIagent-Wallet/1.0',
            'Accept': 'application/json'
        })
        
        # Кэш для адресов токенов
        self.token_address_cache = {}
        self.cache_duration = 30  # 30 секунд для курсов
        
        # Загружаем конфигурацию
        self.config = self._load_config()

        # API endpoints - только 0x (используем v2 API согласно официальной документации)
        self.apis = {
            '0x': {
                'base_url': 'https://api.0x.org',
                'quote_url': '/swap/allowance-holder/quote',
                'price_url': '/swap/allowance-holder/price',
                'tokens_url': '/token/v1',
                'api_key': self.config.get('ZEROX_API_KEY')
            }
        }
        
        # Chain IDs - поддерживаемые 0x Swap API сети
        self.chain_ids = {
            # Основные сети
            'ethereum': 1, 'ETH': 1, 'eth': 1,
            'polygon': 137, 'Polygon': 137,
            'bsc': 56, 'BSC': 56,
            'base': 8453, 'Base': 8453,
            'arbitrum': 42161, 'Arbitrum': 42161,
            'avalanche': 43114, 'Avalanche': 43114,
            'optimism': 10, 'Optimism': 10,
            
            # Дополнительные поддерживаемые сети
            'berachain': 80094, 'Berachain': 80094,
            'blast': 81457, 'Blast': 81457,
            'ink': 57073, 'Ink': 57073,
            'linea': 59144, 'Linea': 59144,
            'mantle': 5000, 'Mantle': 5000,
            'mode': 34443, 'Mode': 34443,
            'monad': 10143, 'Monad': 10143,
            'scroll': 534352, 'Scroll': 534352,
            'unichain': 130, 'Unichain': 130,
            'worldchain': 480, 'World Chain': 480, 'Worldchain': 480,
            
            # Устаревшие сети (не поддерживаются 0x API v2)
            # 'zksync': 324, 'zkSync': 324,  # Не поддерживается
            # 'fantom': 250, 'Fantom': 250,  # Не поддерживается
            # 'cronos': 25, 'Cronos': 25,    # Не поддерживается
            # 'harmony': 1666600000, 'Harmony': 1666600000,  # Не поддерживается
            # 'celo': 42220, 'Celo': 42220,  # Не поддерживается
            # 'gnosis': 100, 'Gnosis': 100,  # Не поддерживается
            # 'fuse': 122, 'Fuse': 122,      # Не поддерживается
            # 'kava': 2222, 'Kava': 2222,    # Не поддерживается
            # 'evmos': 9001, 'Evmos': 9001,  # Не поддерживается
            # 'moonbeam': 1284, 'Moonbeam': 1284,  # Не поддерживается
            # 'moonriver': 1285, 'Moonriver': 1285,  # Не поддерживается
            # 'aurora': 1313161554, 'Aurora': 1313161554,  # Не поддерживается
            # 'metis': 1088, 'Metis': 1088,  # Не поддерживается
            # 'boba': 288, 'Boba': 288       # Не поддерживается
        }
        
        # Token addresses (основные токены) - только самые важные для избежания подвисания
        self.token_addresses = {
            'ethereum': {
                'ETH': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDT': '0xdAC17F958D2ee523a2206206994597C13D831ec7',
                'USDC': '0xA0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                'DAI': '0x6B175474E89094C44Da98b954EedeAC495271d0F'
            },
            'polygon': {
                'POL': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDT': '0xc2132D05D31c914a87C6611C10748AEb04B58e8F',
                'USDC': '0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174',
                'DAI': '0x8f3Cf7ad23Cd3CaDbD9735AFf958023239c6A063',
                'WPOL': '0x0d500B1d8E8eF31E21C99d1Db9A6444d3ADf1270',
                'WETH': '0x7ceB23fD6bC0adD59B62aC2d8A7c36f03125CEfD'
            },
            'bsc': {
                'BNB': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDT': '0x55d398326f99059fF775485246999027B3197955',
                'USDC': '0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d',
                'BUSD': '0xe9e7CEA3DedcA5984780Bafc599bD69ADd087D56'
            },
            'base': {
                'ETH': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDC': '0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913',
                'DAI': '0x50c5725949A6F0c72E6C4a641F24049A917DB0Cb'
            },
            'arbitrum': {
                'ETH': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDT': '0xFd086bC7CD5C481DCC9C85ebE478A1C0b69FCbb9',
                'USDC': '0xaf88d065e77c8cC2239327C5EDb3A432268e5831',
                'ARB': '0x912CE59144191C1204E64559FE8253a0e49E6548'
            },
            'zksync': {
                'ETH': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDC': '0x3355df6D4c9C3035724Fd0e3914dE96A5a83aaf4',
                'USDT': '0x493257fD37EDB34451f62EDf8D2a0C418852bA4C'
            },
            # Алиасы для совместимости
            'ETH': {
                'ETH': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDT': '0xdAC17F958D2ee523a2206206994597C13D831ec7',
                'USDC': '0xA0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                'DAI': '0x6B175474E89094C44Da98b954EedeAC495271d0F'
            },
            'Polygon': {
                'POL': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDT': '0xc2132D05D31c914a87C6611C10748AEb04B58e8F',
                'USDC': '0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174',
                'DAI': '0x8f3Cf7ad23Cd3CaDbD9735AFf958023239c6A063',
                'WPOL': '0x0d500B1d8E8eF31E21C99d1Db9A6444d3ADf1270',
                'WETH': '0x7ceB23fD6bC0adD59B62aC2d8A7c36f03125CEfD'
            },
            'BSC': {
                'BNB': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDT': '0x55d398326f99059fF775485246999027B3197955',
                'USDC': '0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d',
                'BUSD': '0xe9e7CEA3DedcA5984780Bafc599bD69ADd087D56'
            },
            'Base': {
                'ETH': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDC': '0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913',
                'DAI': '0x50c5725949A6F0c72E6C4a641F24049A917DB0Cb'
            },
            'Arbitrum': {
                'ETH': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDT': '0xFd086bC7CD5C481DCC9C85ebE478A1C0b69FCbb9',
                'USDC': '0xaf88d065e77c8cC2239327C5EDb3A432268e5831',
                'ARB': '0x912CE59144191C1204E64559FE8253a0e49E6548'
            },
            'zkSync': {
                'ETH': '0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee',
                'USDC': '0x3355df6D4c9C3035724Fd0e3914dE96A5a83aaf4',
                'USDT': '0x493257fD37EDB34451f62EDf8D2a0C418852bA4C'
            }
        }
        
        # Загружаем API ключи при инициализации
        self._set_api_keys()
    
    def _is_cache_valid(self, cache_key: str) -> bool:
        """Проверяет, действителен ли кэш"""
        if cache_key not in self.token_address_cache:
            return False
        
        import time
        current_time = time.time()
        return (current_time - self.token_address_cache[cache_key]['timestamp']) < self.cache_duration
    
    def _update_cache(self, cache_key: str, value: str):
        """Обновляет кэш"""
        import time
        self.token_address_cache[cache_key] = {
            'value': value,
            'timestamp': time.time()
        }
    
    def _get_from_cache(self, cache_key: str) -> Optional[str]:
        """Получает значение из кэша"""
        if self._is_cache_valid(cache_key):
            return self.token_address_cache[cache_key]['value']
        return None
    
    def _load_config(self) -> Dict[str, str]:
        """Загружает конфигурацию из файла config/RPC.txt."""
        config = {}
        try:
            with open("config/RPC.txt", "r") as f:
                for line in f:
                    line = line.strip()
                    if not line or line.startswith('#'):
                        continue
                    if '=' in line:
                        key, value = line.split('=', 1)
                        config[key.strip()] = value.strip()
        except Exception as e:
            print(f"Ошибка загрузки конфигурации: {e}")
        return config

    def _set_api_keys(self):
        """Устанавливает API ключи из загруженной конфигурации."""
        # Для 0x ключ уже установлен при инициализации self.apis
        zerox_key = self.config.get('ZEROX_API_KEY')
        if zerox_key:
            self.apis['0x']['api_key'] = zerox_key
            # 0x может требовать ключ в заголовке по-другому, например:
            # self.session.headers.update({'0x-api-key': zerox_key})
    
    def get_best_quote(self, from_token: str, to_token: str, amount: float, 
                      chain: str = 'polygon', slippage: float = 0.5, force_refresh: bool = True) -> Optional[Dict[str, Any]]:
        """
        Получает курс обмена через 0x API
        """
        try:
            # Принудительно получаем свежую котировку от 0x
            quote_0x = self._get_0x_quote(from_token, to_token, amount, chain, slippage, force_refresh)
            if quote_0x:
                return {
                    'aggregator': '0x',
                    'fromToken': from_token,
                    'toToken': to_token,
                    'fromTokenAmount': amount,
                    'toTokenAmount': quote_0x.get('toTokenAmount', quote_0x.get('amount_out', 0)),
                    'rate': quote_0x['rate'],
                    'slippage': slippage,
                    'chain': chain
                }
            
            return None
        
        except Exception as e:
            print(f"Ошибка получения котировки: {e}")
            return None
    
    def execute_swap(self, from_token: str, to_token: str, amount: float, 
                    chain: str = 'polygon', private_key: Optional[str] = None) -> Optional[Dict[str, Any]]:
        """
        Выполняет реальный обмен токенов через 0x
        """
        # Инициализируем переменные для возврата результата
        expected_to_token_amount = 0
        expected_rate = 0
        
        try:
            print(f"🔍 Начинаем выполнение свопа: {from_token} -> {to_token}, amount={amount}, chain={chain}")
            
            if not private_key:
                raise Exception("Приватный ключ не предоставлен")
            
            # Получаем котировку с принудительным обновлением
            print(f"🔍 Получаем котировку для {from_token} -> {to_token}")
            quote = self.get_best_quote(from_token, to_token, amount, chain, 0.5, force_refresh=True)
            print(f"🔍 Результат get_best_quote: {quote}")
            if not quote:
                raise Exception("Не удалось получить котировку")
            
            print(f"✅ Котировка получена: {quote}")
            
            # Сохраняем данные котировки для возврата результата
            expected_to_token_amount = quote.get('toTokenAmount', 0)
            expected_rate = quote.get('rate', 0)
            
            # Получаем адрес кошелька из приватного ключа
            from eth_account import Account
            account = Account.from_key(private_key)
            wallet_address = account.address
            print(f"🔍 Адрес кошелька: {wallet_address}")
            
            # Выполняем своп через 0x
            print(f"🔍 Выполняем своп через 0x...")
            result = self._execute_0x_swap(quote, wallet_address, private_key, chain)
            print(f"🔍 Результат свопа: {result}")
            return result
        
        except Exception as e:
            print(f"❌ Ошибка выполнения свопа: {e}")
            import traceback
            traceback.print_exc()
            return {'success': False, 'error': str(e)}
    
    
    def _execute_0x_swap(self, quote: Dict[str, Any], wallet_address: str, 
                        private_key: str, chain: str) -> Optional[Dict[str, Any]]:
        """Выполняет своп через 0x Protocol"""
        try:
            chain_id = self.chain_ids.get(chain, 1)
            from_address = self._get_token_address(quote['fromToken'], chain)
            to_address = self._get_token_address(quote['toToken'], chain)
            
            if not from_address or not to_address:
                print(f"❌ Не удалось получить адреса токенов: from={from_address}, to={to_address}")
                return None
            
            # Получаем данные для свопа через правильный quote endpoint v2
            url = f"{self.apis['0x']['base_url']}/swap/allowance-holder/quote"
            # Используем правильные decimals для sellAmount
            from_token = quote['fromToken']
            if from_token.upper() in ['ETH', 'POL', 'BNB']:
                sell_amount_wei = Web3.to_wei(quote['fromTokenAmount'], 'ether')
            else:
                from_decimals = self._get_token_decimals(from_address, from_token)
                sell_amount_wei = int(quote['fromTokenAmount'] * (10 ** from_decimals))
            
            params = {
                'chainId': str(self.chain_ids.get(chain, 137)),
                'buyToken': to_address,
                'sellToken': from_address,
                'sellAmount': str(sell_amount_wei),
                'taker': wallet_address,
                'timestamp': str(int(time.time() * 1000))  # Принудительное обновление
            }
            
            print(f"🔍 Выполняем своп: {from_token} -> {quote['toToken']}, amount={quote['fromTokenAmount']}, wei={sell_amount_wei}")
            
            # Заголовки согласно документации v2 с принудительным обновлением
            headers = {
                'Content-Type': 'application/json',
                '0x-api-key': self.apis['0x']['api_key'],
                '0x-version': 'v2',
                'Cache-Control': 'no-cache'
            }
            
            response = self.session.get(url, params=params, headers=headers, timeout=20)
            
            if response.status_code == 200:
                swap_data = response.json()
                print(f"🔍 0x API ответ: {swap_data}")
                
                # Подписываем и отправляем транзакцию
                return self._sign_and_send_transaction(swap_data, private_key, chain)
            else:
                print(f"❌ Ошибка 0x API: {response.status_code} - {response.text}")
                return {'success': False, 'error': f"0x API error: {response.status_code} - {response.text}"}
        
        except Exception as e:
            print(f"Ошибка 0x свопа: {e}")
            import traceback
            traceback.print_exc()
            return {'success': False, 'error': str(e)}
    
    def _sign_and_send_transaction(self, tx_data: Dict[str, Any], private_key: str, 
                                 chain: str) -> Optional[Dict[str, Any]]:
        """Подписывает и отправляет транзакцию"""
        try:
            from web3 import Web3
            from eth_account import Account
            
            # Подключаемся к сети, используя централизованный конфиг
            rpc_url = self.config.get(f"{chain.upper()}_RPC_URL")
            if not rpc_url:
                raise ValueError(f"RPC URL для сети {chain} не найден в конфигурации.")

            web3 = Web3(Web3.HTTPProvider(rpc_url))
            
            # Создаем аккаунт
            account = Account.from_key(private_key)
            
            # Подготавливаем транзакцию из данных 0x API
            # 0x API v2 может возвращать данные в разных полях
            transaction_data = tx_data.get('tx') or tx_data.get('transaction') or tx_data
            to_address = transaction_data.get('to') or transaction_data.get('target')
            
            if not to_address:
                print(f"🔍 Доступные поля в ответе 0x API: {list(tx_data.keys())}")
                if 'tx' in tx_data:
                    print(f"🔍 Содержимое поля 'tx': {tx_data['tx']}")
                if 'transaction' in tx_data:
                    print(f"🔍 Содержимое поля 'transaction': {tx_data['transaction']}")
                raise ValueError("Отсутствует адрес получателя в данных транзакции")
                
            # Безопасно обрабатываем value
            value_hex = transaction_data.get('value', '0x0')
            if value_hex == '0x0' or value_hex == '0x':
                value = 0
            else:
                value = int(value_hex, 16)
                
            # Используем реальные данные от 0x API
            gas_hex = transaction_data.get('gas', '0x186a0')
            gas_limit = int(gas_hex, 16)
            
            # Получаем правильный nonce (учитываем pending транзакции)
            try:
                # Сначала пробуем получить nonce с учетом pending транзакций
                nonce = web3.eth.get_transaction_count(account.address, 'pending')
                print(f"🔍 Nonce (pending): {nonce}")
            except Exception as e:
                print(f"⚠️ Ошибка получения pending nonce, используем latest: {e}")
                nonce = web3.eth.get_transaction_count(account.address, 'latest')
                print(f"🔍 Nonce (latest): {nonce}")
            
            # Оптимизируем gas limit для Polygon (0x API часто завышает)
            if chain.lower() == 'polygon':
                if gas_limit > 1000000:  # Если больше 1M gas
                    gas_limit = 500000  # Используем больше для сложных маршрутов
                    print(f"🔧 Оптимизируем gas limit: {gas_limit} (было больше 1M)")
                elif gas_limit > 500000:  # Если больше 500K gas
                    gas_limit = 400000  # Увеличиваем для сложных маршрутов
                    print(f"🔧 Оптимизируем gas limit: {gas_limit} (было больше 500K)")
                elif gas_limit < 100000:  # Если меньше 100K gas
                    gas_limit = 200000  # Минимум для обмена
                    print(f"🔧 Увеличиваем gas limit до минимума: {gas_limit}")
                else:
                    # Для нормальных значений добавляем буфер (20%)
                    gas_limit = int(gas_limit * 1.2)
                    print(f"🔍 Используем gas limit от 0x API: {gas_limit} (с 20% буфером)")
            else:
                # Для других сетей используем как есть с буфером
                gas_limit = int(gas_limit * 1.2)
                print(f"🔍 Используем gas limit от 0x API: {gas_limit} (с 20% буфером)")
            
            # Для Polygon используем EIP-1559 транзакции (более эффективно)
            if chain.lower() == 'polygon':
                try:
                    # Получаем fee data для EIP-1559
                    fee_data = web3.eth.fee_history(1, 'latest', [50])
                    base_fee = fee_data['baseFeePerGas'][0]
                    max_priority_fee = min(web3.eth.max_priority_fee, 30000000000)  # 30 gwei max
                    max_fee = base_fee * 2 + max_priority_fee
                    
                    transaction = {
                        'to': Web3.to_checksum_address(to_address),
                        'value': value,
                        'data': transaction_data.get('data', '0x'),
                        'gas': gas_limit,
                        'maxFeePerGas': max_fee,
                        'maxPriorityFeePerGas': max_priority_fee,
                        'nonce': nonce,
                        'chainId': web3.eth.chain_id,
                        'type': 2  # EIP-1559
                    }
                    print(f"🔍 EIP-1559 транзакция: maxFee={max_fee}, maxPriorityFee={max_priority_fee}")
                except Exception as e:
                    print(f"⚠️ Ошибка EIP-1559, используем legacy: {e}")
                    # Fallback к legacy транзакции
                    gas_price = min(web3.eth.gas_price, 50000000000)  # 50 gwei max
                    transaction = {
                        'to': Web3.to_checksum_address(to_address),
                        'value': value,
                        'data': transaction_data.get('data', '0x'),
                        'gas': gas_limit,
                        'gasPrice': gas_price,
                        'nonce': nonce,
                        'chainId': web3.eth.chain_id
                    }
            else:
                # Для других сетей используем legacy транзакции
                gas_price = min(web3.eth.gas_price, 50000000000)  # 50 gwei max
                transaction = {
                    'to': Web3.to_checksum_address(to_address),
                    'value': value,
                    'data': transaction_data.get('data', '0x'),
                    'gas': gas_limit,
                    'gasPrice': gas_price,
                    'nonce': nonce,
                    'chainId': web3.eth.chain_id
                }
            
            print(f"🔍 Параметры транзакции: gas={gas_limit}, value={value}")
            
            # Подписываем транзакцию
            signed_txn = web3.eth.account.sign_transaction(transaction, private_key)
            
            # Отправляем транзакцию
            try:
                tx_hash = web3.eth.send_raw_transaction(signed_txn.raw_transaction)
                print(f"🔍 Транзакция отправлена: {tx_hash.hex()}")
                
                # Ждем подтверждения с более коротким таймаутом
                print(f"⏳ Ожидаем подтверждения транзакции...")
                try:
                    receipt = web3.eth.wait_for_transaction_receipt(tx_hash, timeout=120)  # 120 секунд для Polygon
                    print(f"🔍 Транзакция подтверждена: {receipt}")
                    
                    return {
                        'success': receipt['status'] == 1,
                        'tx_hash': tx_hash.hex(),
                        'gas_used': receipt['gasUsed'],
                        'block_number': receipt['blockNumber'],
                        'toTokenAmount': 0,  # Временно используем 0, будет исправлено позже
                        'rate': 0  # Временно используем 0, будет исправлено позже
                    }
                except Exception as timeout_error:
                    print(f"⚠️ Таймаут ожидания подтверждения: {timeout_error}")
                    print(f"🔍 Проверяем статус транзакции альтернативным способом...")
                    
                    # Альтернативная проверка - просто возвращаем успех, если транзакция отправлена
                    # Пользователь может проверить статус в блокчейн сканере
                    return {
                        'success': True,  # Считаем успешным, если транзакция отправлена
                        'tx_hash': tx_hash.hex(),
                        'gas_used': 0,  # Неизвестно
                        'block_number': 0,  # Неизвестно
                        'toTokenAmount': 0,  # Временно используем 0, будет исправлено позже
                        'rate': 0,  # Временно используем 0, будет исправлено позже
                        'note': 'Транзакция отправлена, проверьте статус в блокчейн сканере'
                    }
                    
            except Exception as tx_error:
                error_msg = str(tx_error)
                print(f"❌ Ошибка отправки транзакции: {error_msg}")
                
                # Если ошибка связана с nonce, пробуем с новым nonce
                if 'nonce' in error_msg.lower() or 'replace' in error_msg.lower():
                    print("🔄 Пробуем с новым nonce...")
                    try:
                        # Получаем свежий nonce
                        new_nonce = web3.eth.get_transaction_count(account.address, 'pending')
                        print(f"🔍 Новый nonce: {new_nonce}")
                        
                        # Обновляем nonce в транзакции
                        transaction['nonce'] = new_nonce
                        
                        # Подписываем заново
                        signed_txn = web3.eth.account.sign_transaction(transaction, private_key)
                        
                        # Отправляем с новым nonce
                        tx_hash = web3.eth.send_raw_transaction(signed_txn.raw_transaction)
                        print(f"🔍 Транзакция отправлена с новым nonce: {tx_hash.hex()}")
                        
                        # Ждем подтверждения с коротким таймаутом
                        print(f"⏳ Ожидаем подтверждения транзакции...")
                        receipt = web3.eth.wait_for_transaction_receipt(tx_hash, timeout=60)
                        print(f"🔍 Транзакция подтверждена: {receipt}")
                        
                        return {
                            'success': receipt['status'] == 1,
                            'tx_hash': tx_hash.hex(),
                            'gas_used': receipt['gasUsed'],
                            'block_number': receipt['blockNumber']
                        }
                        
                    except Exception as retry_error:
                        print(f"❌ Ошибка при повторной попытке: {retry_error}")
                        return {
                            'success': False,
                            'error': f"Ошибка nonce: {error_msg}"
                        }
                else:
                    return {
                        'success': False,
                        'error': f"Ошибка транзакции: {error_msg}"
                    }
        
        except ValueError as e:
            print(f"Ошибка валидации данных транзакции: {e}")
            return {'success': False, 'error': str(e)}
        except Exception as e:
            print(f"Ошибка подписания/отправки транзакции: {e}")
            print(f"Данные транзакции: {tx_data}")
            return {'success': False, 'error': str(e)}

    def _get_rpc_url(self, chain: str) -> str:
        """Получает RPC URL для сети из конфигурации."""
        # Сначала пробуем найти в конфигурации
        rpc_url = self.config.get(f"{chain.upper()}_RPC_URL")
        if rpc_url:
            return rpc_url
        
        # Fallback RPC URLs для основных сетей
        fallback_rpcs = {
            'ethereum': 'https://rpc.ankr.com/eth/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'polygon': 'https://rpc.ankr.com/polygon/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'bsc': 'https://rpc.ankr.com/bsc/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'base': 'https://rpc.ankr.com/base/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'arbitrum': 'https://rpc.ankr.com/arbitrum/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'zksync': 'https://mainnet.era.zksync.io',
            'avalanche': 'https://rpc.ankr.com/avalanche/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'optimism': 'https://mainnet.optimism.io',
            'fantom': 'https://rpc.ftm.tools',
            'cronos': 'https://evm.cronos.org',
            'harmony': 'https://api.harmony.one',
            'celo': 'https://forno.celo.org',
            'gnosis': 'https://rpc.gnosischain.com',
            'fuse': 'https://rpc.fuse.io',
            'kava': 'https://evm.kava.io',
            'evmos': 'https://eth.bd.evmos.org:8545',
            'moonbeam': 'https://rpc.api.moonbeam.network',
            'moonriver': 'https://rpc.api.moonriver.moonbeam.network',
            'aurora': 'https://mainnet.aurora.dev',
            'metis': 'https://andromeda.metis.io/?owner=1088',
            'boba': 'https://mainnet.boba.network',
            # Алиасы
            'ETH': 'https://rpc.ankr.com/eth/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'Polygon': 'https://rpc.ankr.com/polygon/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'BSC': 'https://rpc.ankr.com/bsc/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'Base': 'https://rpc.ankr.com/base/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'Arbitrum': 'https://rpc.ankr.com/arbitrum/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'zkSync': 'https://mainnet.era.zksync.io',
            'Avalanche': 'https://rpc.ankr.com/avalanche/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7',
            'Optimism': 'https://mainnet.optimism.io',
            'Fantom': 'https://rpc.ftm.tools',
            'Cronos': 'https://evm.cronos.org',
            'Harmony': 'https://api.harmony.one',
            'Celo': 'https://forno.celo.org',
            'Gnosis': 'https://rpc.gnosischain.com',
            'Fuse': 'https://rpc.fuse.io',
            'Kava': 'https://evm.kava.io',
            'Evmos': 'https://eth.bd.evmos.org:8545',
            'Moonbeam': 'https://rpc.api.moonbeam.network',
            'Moonriver': 'https://rpc.api.moonriver.moonbeam.network',
            'Aurora': 'https://mainnet.aurora.dev',
            'Metis': 'https://andromeda.metis.io/?owner=1088',
            'Boba': 'https://mainnet.boba.network'
        }
        
        rpc_url = fallback_rpcs.get(chain.lower()) or fallback_rpcs.get(chain)
        if rpc_url:
            print(f"🔍 Используем fallback RPC для {chain}: {rpc_url}")
            return rpc_url
        
        print(f"❌ RPC для сети {chain} не найден. Используется Ethereum fallback.")
        return "https://rpc.ankr.com/eth/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7"
    
    def get_swap_history(self, wallet_address: str, chain: str = 'ETH', 
                        limit: int = 10) -> List[Dict[str, Any]]:
        """Получает историю свопов для адреса"""
        try:
            # Здесь можно добавить интеграцию с Etherscan API или другими сервисами
            # Пока возвращаем заглушку
            return [
                {
                    'tx_hash': '0x123...',
                    'from_token': 'ETH',
                    'to_token': 'USDT',
                    'amount': 0.1,
                    'received': 185.0,
                    'timestamp': int(time.time()) - 3600,
                    'status': 'success'
                }
            ]
        
        except Exception as e:
            print(f"Ошибка получения истории свопов: {e}")
            return []
    
    def get_token_allowance(self, token_address: str, wallet_address: str, 
                           spender_address: str, chain: str = 'ETH') -> float:
        """Проверяет allowance токена для DEX"""
        try:
            from web3 import Web3
            
            rpc_url = self._get_rpc_url(chain)
            web3 = Web3(Web3.HTTPProvider(rpc_url))
            
            # ABI для allowance
            allowance_abi = [
                {
                    "constant": True,
                    "inputs": [
                        {"name": "_owner", "type": "address"},
                        {"name": "_spender", "type": "address"}
                    ],
                    "name": "allowance",
                    "outputs": [{"name": "", "type": "uint256"}],
                    "type": "function"
                }
            ]
            
            token_contract = web3.eth.contract(
                address=Web3.to_checksum_address(token_address),
                abi=allowance_abi
            )
            
            allowance = token_contract.functions.allowance(
                Web3.to_checksum_address(wallet_address),
                Web3.to_checksum_address(spender_address)
            ).call()
            
            print(f"🔍 Raw allowance: {allowance}")
            
            # USDT имеет 6 десятичных знаков, другие токены - 18
            if token_address.lower() == '0xc2132d05d31c914a87c6611c10748aeb04b58e8f':  # USDT на Polygon
                decimals = 6
            else:
                decimals = 18
            
            result = float(allowance) / (10 ** decimals)
            print(f"🔍 Calculated allowance: {result} (decimals: {decimals})")
            return result
        
        except Exception as e:
            print(f"❌ Ошибка проверки allowance: {e}")
            print(f"🔍 Параметры: token={token_address}, wallet={wallet_address}, spender={spender_address}, chain={chain}")
            return 0.0
    
    def approve_token(self, token_address: str, spender_address: str, 
                     private_key: str, chain: str = 'ETH', amount: str = '0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff') -> Optional[Dict[str, Any]]:
        """Выдает разрешение на использование токенов DEX"""
        try:
            from web3 import Web3
            from eth_account import Account
            
            rpc_url = self._get_rpc_url(chain)
            web3 = Web3(Web3.HTTPProvider(rpc_url))
            
            # ABI для approve
            approve_abi = [
                {
                    "constant": False,
                    "inputs": [
                        {"name": "_spender", "type": "address"},
                        {"name": "_value", "type": "uint256"}
                    ],
                    "name": "approve",
                    "outputs": [{"name": "", "type": "bool"}],
                    "type": "function"
                }
            ]
            
            token_contract = web3.eth.contract(
                address=Web3.to_checksum_address(token_address),
                abi=approve_abi
            )
            
            account = Account.from_key(private_key)
            
            # Создаем транзакцию approve
            transaction = token_contract.functions.approve(
                Web3.to_checksum_address(spender_address),
                int(amount, 16)
            ).build_transaction({
                'from': account.address,
                'gas': 100000,
                'gasPrice': web3.eth.gas_price,
                'nonce': web3.eth.get_transaction_count(account.address),
                'chainId': web3.eth.chain_id
            })
            
            # Подписываем и отправляем
            signed_txn = web3.eth.account.sign_transaction(transaction, private_key)
            tx_hash = web3.eth.send_raw_transaction(signed_txn.raw_transaction)
            
            # Ждем подтверждения
            receipt = web3.eth.wait_for_transaction_receipt(tx_hash, timeout=300)
            
            return {
                'success': receipt['status'] == 1,
                'tx_hash': tx_hash.hex(),
                'gas_used': receipt['gasUsed']
            }
        
        except Exception as e:
            print(f"Ошибка approve токена: {e}")
            return None
    
    
    def _get_0x_quote(self, from_token: str, to_token: str, amount: float, 
                     chain: str, slippage: float, force_refresh: bool = True,
                     from_token_info: Optional[Dict] = None, to_token_info: Optional[Dict] = None) -> Optional[Dict[str, Any]]:
        """Получает котировку от 0x Protocol согласно официальной документации"""
        try:
            chain_id = self.chain_ids.get(chain, 137)  # По умолчанию Polygon
            
            # Используем переданную информацию о токенах, если доступна
            if from_token_info and 'address' in from_token_info:
                from_address = from_token_info['address']
                print(f"✅ Используем адрес from_token из переданной информации: {from_address}")
            else:
                from_address = self._get_token_address(from_token, chain)
            
            if to_token_info and 'address' in to_token_info:
                to_address = to_token_info['address']
                print(f"✅ Используем адрес to_token из переданной информации: {to_address}")
            else:
                to_address = self._get_token_address(to_token, chain)
            
            if not from_address or not to_address:
                print(f"❌ Не удалось найти адреса токенов: from={from_address}, to={to_address}")
                return None
            
            # Конвертируем в wei согласно документации
            if from_token.upper() in ['ETH', 'BNB']:
                amount_wei = Web3.to_wei(amount, 'ether')
            elif from_token.upper() == 'POL':
                # Для POL на Polygon используем WPOL
                from_address = self.token_addresses.get(chain, {}).get('WPOL', from_address)
                amount_wei = Web3.to_wei(amount, 'ether')
            else:
                # Используем правильные decimals для токена
                if from_token_info and 'decimals' in from_token_info:
                    from_decimals = from_token_info['decimals']
                    print(f"✅ Используем decimals из переданной информации: {from_decimals}")
                else:
                    from_decimals = self._get_token_decimals(from_address, from_token)
                amount_wei = int(amount * (10 ** from_decimals))
            
            # Используем /price endpoint для получения котировок согласно документации v2
            url = f"{self.apis['0x']['base_url']}/swap/allowance-holder/price"
            params = {
                'chainId': str(chain_id),
                'buyToken': to_address,
                'sellToken': from_address,
                'sellAmount': str(amount_wei),
                'taker': '0xa89E857ae6eD1C145F81d4c9eBF409B855c73820'  # Реальный адрес кошелька
            }
            
            # Заголовки согласно документации v2
            headers = {
                'Content-Type': 'application/json',
                '0x-api-key': self.apis['0x']['api_key'],
                '0x-version': 'v2',
                'Cache-Control': 'no-cache' if force_refresh else 'max-age=30'
            }
            
            # Добавляем timestamp для принудительного обновления
            if force_refresh:
                params['timestamp'] = str(int(time.time() * 1000))
            
            print(f"🔍 Запрашиваем цену от 0x: {url} с параметрами: {params}")
            response = self.session.get(url, params=params, headers=headers, timeout=20)
            
            if response.status_code == 200:
                data = response.json()
                print(f"✅ 0x Price API ответ: {data}")
                
                # Проверяем доступность ликвидности
                if not data.get('liquidityAvailable', True):
                    print(f"❌ Ликвидность недоступна для {from_token} -> {to_token}")
                    # Пробуем альтернативные пары токенов
                    if from_token.upper() == 'ETH' and to_token.upper() == 'USDT':
                        print(f"🔄 Пробуем альтернативную пару: WETH -> USDT")
                        # Пробуем WETH вместо ETH
                        weth_address = self.token_addresses.get(chain, {}).get('WETH')
                        if weth_address:
                            return self._get_0x_quote('WETH', to_token, amount, chain, slippage, force_refresh)
                    return None
                
                # Парсим ответ согласно структуре /price API v2
                buy_amount_raw = data.get('buyAmount', '0')
                sell_amount_raw = data.get('sellAmount', '0')
                estimated_gas = int(data.get('gas', 0))
                
                # Проверяем, что получили валидные суммы
                if buy_amount_raw == '0' or sell_amount_raw == '0':
                    print(f"❌ Нулевые суммы от 0x API: buy={buy_amount_raw}, sell={sell_amount_raw}")
                    return None
                
                # Конвертируем из базовых единиц в токены
                buy_decimals = self._get_token_decimals(to_address, to_token)
                sell_decimals = self._get_token_decimals(from_address, from_token)
                
                buy_amount = float(buy_amount_raw) / (10 ** buy_decimals)
                sell_amount = float(sell_amount_raw) / (10 ** sell_decimals)
                
                # Вычисляем курс обмена
                rate = buy_amount / sell_amount if sell_amount > 0 else 0
                
                print(f"🔍 Парсинг 0x v2 ответа: buyAmount={buy_amount_raw}, sellAmount={sell_amount_raw}")
                print(f"🔍 Конвертированные суммы: buy={buy_amount:.6f}, sell={sell_amount:.6f}, rate={rate:.6f}")
                
                # Исправляем символы токенов (0x API иногда возвращает неправильные символы)
                corrected_from_token = from_token
                corrected_to_token = to_token
                
                # Проверяем, есть ли исправления в ответе API
                if 'tokens' in data:
                    for token_info in data['tokens']:
                        token_address = token_info.get('address', '').lower()
                        api_symbol = token_info.get('symbol', '')
                        
                        # Если адрес совпадает, но символ неправильный, исправляем
                        if token_address == from_address.lower() and api_symbol != from_token:
                            print(f"🔧 Исправляем символ токена: {api_symbol} -> {from_token}")
                            corrected_from_token = from_token
                        
                        if token_address == to_address.lower() and api_symbol != to_token:
                            print(f"🔧 Исправляем символ токена: {api_symbol} -> {to_token}")
                            corrected_to_token = to_token
                
                # Дополнительная проверка на известные проблемы с символами
                if corrected_from_token == "USDT0":
                    corrected_from_token = "USDT"
                    print(f"🔧 Исправляем известную проблему: USDT0 -> USDT")
                if corrected_to_token == "USDT0":
                    corrected_to_token = "USDT"
                    print(f"🔧 Исправляем известную проблему: USDT0 -> USDT")
                
                # Исправляем WPOL -> POL
                if corrected_from_token == "WPOL":
                    corrected_from_token = "POL"
                    print(f"🔧 Исправляем символ токена: WPOL -> POL")
                if corrected_to_token == "WPOL":
                    corrected_to_token = "POL"
                    print(f"🔧 Исправляем символ токена: WPOL -> POL")
                
                return {
                    'fromTokenAmount': sell_amount,
                    'toTokenAmount': buy_amount,
                    'amount_out': buy_amount,  # Добавляем для совместимости
                    'rate': rate,  # Вычисляем курс из buyAmount/sellAmount
                    'estimatedGas': estimated_gas,
                    'protocols': data.get('route', {}).get('fills', []),
                    'gas': estimated_gas,
                    'fromToken': corrected_from_token,
                    'toToken': corrected_to_token,
                    'chain': chain,
                    'provider': '0x',
                    'spender_address': data.get('allowanceTarget', '0xDef1C0ded9bec7F1a1670819833240f027b25EfF'),  # 0x Protocol Router
                    'data': data  # Сохраняем полные данные для выполнения свопа
                }
            
            print(f"❌ 0x Price API error: {response.status_code}, {response.text}")
            return None
        
        except Exception as e:
            print(f"Ошибка 0x API: {e}")
            return None
    
    def _get_token_decimals(self, token_address: str, token_symbol: str) -> int:
        """Получает количество decimals для токена"""
        # Проверяем предустановленные токены
        for chain_tokens in PREDEFINED_TOKENS.values():
            for token_info in chain_tokens.values():
                if token_info.get('address', '').lower() == token_address.lower():
                    return token_info.get('decimals', 18)
        
        # Fallback на основе символа
        if token_symbol.upper() in ['USDC', 'USDT']:
            return 6
        elif token_symbol.upper() in ['POL', 'ETH', 'BNB', 'WETH', 'WPOL']:
            return 18
        else:
            return 18  # По умолчанию
    
    def _get_token_address(self, token: str, chain: str) -> Optional[str]:
        """Получает адрес токена для указанной сети - использует PREDEFINED_TOKENS"""
        try:
            # Получаем chain_id для сети
            chain_id = self.chain_ids.get(chain.lower())
            if not chain_id:
                print(f"❌ Неизвестная сеть: {chain}")
                return None
            
            # Проверяем в PREDEFINED_TOKENS
            if chain_id in PREDEFINED_TOKENS:
                tokens = PREDEFINED_TOKENS[chain_id]
                token_upper = token.upper()
                
                # Ищем точное совпадение
                if token_upper in tokens:
                    address = tokens[token_upper].get('address')
                    print(f"✅ Найден адрес токена {token} в {chain}: {address}")
                    return address
                
                # Ищем частичное совпадение
                for symbol, token_info in tokens.items():
                    if token_upper in symbol.upper():
                        address = token_info.get('address')
                        print(f"✅ Найден адрес токена {token} (как {symbol}) в {chain}: {address}")
                        return address
            
            print(f"❌ Токен {token} не найден в сети {chain}")
            return None
        
        except Exception as e:
            print(f"Ошибка получения адреса токена: {e}")
            return None
    
    
    def get_gas_estimate(self, chain: str = 'ETH') -> Optional[Dict[str, Any]]:
        """Получает оценку газа для сети"""
        try:
            if chain == 'ETH':
                # Используем Infura Gas API
                url = "https://gas.api.infura.io/v3/524f929431e44c16a21220aa90e6b253"
                response = self.session.get(url, timeout=20)
                
                if response.status_code == 200:
                    data = response.json()
                    return {
                        'slow': data.get('slow', {}).get('suggestedMaxFeePerGas', 0),
                        'standard': data.get('standard', {}).get('suggestedMaxFeePerGas', 0),
                        'fast': data.get('fast', {}).get('suggestedMaxFeePerGas', 0)
                    }
            
            return None
        
        except Exception as e:
            print(f"Ошибка получения газа: {e}")
            return None

    def get_available_tokens(self, chain: str) -> Dict[str, Dict[str, Any]]:
        """Получает доступные токены для сети (0x API не предоставляет эндпоинт для токенов)"""
        try:
            print(f"🔍 0x API не предоставляет эндпоинт для получения токенов")
            print(f"🔍 Используем предустановленные токены для сети {chain}")
            
            # 0x API не предоставляет эндпоинт для получения токенов
            # Используем предустановленные токены
            chain_id = self.chain_ids.get(chain, 137)
            if chain_id in PREDEFINED_TOKENS:
                tokens = {}
                for symbol, token_data in PREDEFINED_TOKENS[chain_id].items():
                    tokens[symbol] = {
                        'address': token_data['address'],
                        'decimals': token_data['decimals'],
                        'symbol': symbol
                    }
                print(f"✅ Найдено {len(tokens)} предустановленных токенов для {chain}")
                return tokens
            else:
                print(f"❌ Нет предустановленных токенов для сети {chain}")
                return {}
                
        except Exception as e:
            print(f"❌ Ошибка при получении токенов: {e}")
            return {}

    def get_available_networks(self) -> Dict[str, Dict[str, Any]]:
        """Получает доступные сети от 0x API"""
        try:
            # URL для получения сетей согласно документации
            url = f"https://api.0x.org/swap/chains"
            
            # Заголовки
            headers = {
                'Content-Type': 'application/json',
                '0x-api-key': self.apis['0x']['api_key'],
                '0x-version': 'v2'
            }
            
            print(f"🔍 Запрашиваем сети от 0x API")
            
            response = requests.get(url, headers=headers, timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                print(f"✅ 0x Chains API ответ: {data}")
                
                # Проверяем тип данных
                if isinstance(data, dict) and 'chains' in data:
                    # API возвращает объект с ключом 'chains'
                    chains = data['chains']
                    networks = {}
                    for network in chains:
                        if isinstance(network, dict):
                            chain_id = network.get('chainId')
                            chain_name = network.get('chainName', 'Unknown')
                            
                            if chain_id:
                                networks[chain_name] = {
                                    'chainId': int(chain_id),
                                    'name': chain_name,
                                    'rpcUrl': '',  # 0x API не предоставляет RPC URLs
                                    'explorerUrl': ''  # 0x API не предоставляет explorer URLs
                                }
                    return networks
                elif isinstance(data, list):
                    # Прямой массив сетей
                    networks = {}
                    for network in data:
                        if isinstance(network, dict):
                            chain_id = network.get('chainId')
                            chain_name = network.get('chainName', 'Unknown')
                            
                            if chain_id:
                                networks[chain_name] = {
                                    'chainId': int(chain_id),
                                    'name': chain_name,
                                    'rpcUrl': '',  # 0x API не предоставляет RPC URLs
                                    'explorerUrl': ''  # 0x API не предоставляет explorer URLs
                                }
                    return networks
                else:
                    print(f"❌ Неожиданный формат ответа: {type(data)}")
                    return {}
            else:
                print(f"❌ Ошибка 0x Chains API: {response.status_code} - {response.text}")
                return {}
                
        except Exception as e:
            print(f"❌ Ошибка при получении сетей от 0x: {e}")
            return {}
    
    def get_quote(self, from_token: str, to_token: str, amount: int, network: str) -> Optional[Dict[str, Any]]:
        """
        Получает котировку для обмена токенов
        
        Args:
            from_token: Символ токена для продажи
            to_token: Символ токена для покупки  
            amount: Количество токенов для продажи (в wei)
            network: Название сети
            
        Returns:
            Словарь с информацией о котировке или None
        """
        try:
            print(f"🔍 Получаем котировку {from_token} -> {to_token} на {network}")
            
            # Получаем chain_id
            chain_id = self.chain_ids.get(network.lower())
            if not chain_id:
                print(f"❌ Неизвестная сеть: {network}")
                return None
            
            # Получаем адреса токенов
            from_address = self._get_token_address(from_token, network)
            to_address = self._get_token_address(to_token, network)
            
            if not from_address or not to_address:
                print(f"❌ Не удалось найти адреса токенов: {from_token} -> {to_token}")
                return None
            
            # Используем 0x API для получения котировки
            quote = self._get_0x_quote(from_token, to_token, amount, network, 1.0)
            
            if quote:
                print(f"✅ Котировка получена: {quote.get('fromAmount')} {from_token} -> {quote.get('toAmount')} {to_token}")
                return quote
            else:
                print(f"❌ Котировка не найдена для {from_token} -> {to_token}")
                return None
                
        except Exception as e:
            print(f"❌ Ошибка получения котировки: {e}")
            return None


class Plugin:
    """
    Плагин для интеграции с DEX агрегаторами.
    """
    def __init__(self, main_window):
        self.main_window = main_window
        self.dependencies = ["requests"]
    
    def get_instance(self):
        """Возвращает экземпляр DEX интеграции."""
        return DEXIntegration()
