"""
Модуль шифрования для API ключей
Использует Fernet (симметричное шифрование) для защиты API ключей
Автоматически генерирует мастер-ключ для новых пользователей
"""
import base64
import os
import hashlib
import platform
from cryptography.fernet import Fernet
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
import getpass

class APIKeyEncryption:
    """Класс для шифрования/дешифрования API ключей"""
    
    def __init__(self):
        self.key_file = "config/.encryption_key"
        self.salt_file = "config/.salt"
        self._fernet = None
        self._initialize_encryption()
    
    def _generate_system_based_key(self) -> str:
        """Генерирует мастер-ключ на основе системной информации"""
        try:
            # Собираем системную информацию для генерации уникального ключа
            system_info = [
                platform.node(),           # Имя компьютера
                platform.system(),         # ОС (Linux, Windows, Darwin)
                platform.machine(),        # Архитектура (x86_64, arm64)
                platform.processor(),      # Процессор
                str(os.getuid()),         # ID пользователя
                str(os.getgid()),         # ID группы
            ]
            
            # Создаем уникальную строку
            unique_string = ":".join(filter(None, system_info))
            
            # Генерируем хеш
            master_key = hashlib.sha256(unique_string.encode('utf-8')).hexdigest()
            
            # Берем первые 32 символа для стабильности
            return master_key[:32]
            
        except Exception as e:
            print(f"⚠️ Ошибка генерации системного ключа: {e}")
            # Fallback к стабильному ключу
            return "maxport_aiagent_default_key_2024"
    
    def _initialize_encryption(self):
        """Инициализирует шифрование, создавая или загружая ключ"""
        try:
            if not os.path.exists(self.key_file) or not os.path.exists(self.salt_file):
                # Создаем новый ключ шифрования автоматически
                self._create_new_key_auto()
            else:
                # Загружаем существующий ключ
                self._load_existing_key()
        except Exception as e:
            print(f"❌ Ошибка инициализации шифрования: {e}")
            # Создаем новый ключ в случае ошибки
            self._create_new_key_auto()
    
    def _create_new_key_auto(self):
        """Создает новый ключ шифрования автоматически"""
        try:
            # Генерируем мастер-ключ на основе системы
            master_password = self._generate_system_based_key()
            
            # Генерируем соль
            salt = os.urandom(16)
            
            # Создаем ключ из пароля
            kdf = PBKDF2HMAC(
                algorithm=hashes.SHA256(),
                length=32,
                salt=salt,
                iterations=100000,
            )
            key = base64.urlsafe_b64encode(kdf.derive(master_password.encode()))
            
            # Сохраняем ключ и соль
            os.makedirs("config", exist_ok=True)
            with open(self.key_file, "wb") as f:
                f.write(key)
            with open(self.salt_file, "wb") as f:
                f.write(salt)
            
            self._fernet = Fernet(key)
            print("✅ Новый ключ шифрования создан автоматически")
            print("💡 Ключ основан на системной информации, пароль вводить не нужно")
            
        except Exception as e:
            print(f"❌ Ошибка создания ключа шифрования: {e}")
            # Создаем временный ключ для работы
            key = Fernet.generate_key()
            self._fernet = Fernet(key)
    
    def _load_existing_key(self):
        """Загружает существующий ключ шифрования"""
        try:
            # Сначала пытаемся использовать системный ключ
            master_password = self._generate_system_based_key()
            
            try:
                # Загружаем соль
                with open(self.salt_file, "rb") as f:
                    salt = f.read()
                
                # Воссоздаем ключ из системного пароля
                kdf = PBKDF2HMAC(
                    algorithm=hashes.SHA256(),
                    length=32,
                    salt=salt,
                    iterations=100000,
                )
                key = base64.urlsafe_b64encode(kdf.derive(master_password.encode()))
                
                # Загружаем сохраненный ключ
                with open(self.key_file, "rb") as f:
                    stored_key = f.read()
                
                if key == stored_key:
                    self._fernet = Fernet(key)
                    print("✅ Ключ шифрования загружен автоматически")
                    return
            except Exception:
                pass
            
            # Если системный ключ не подошел, пробуем пароль по умолчанию
            try:
                master_password = "default_password_123"
                
                # Загружаем соль
                with open(self.salt_file, "rb") as f:
                    salt = f.read()
                
                # Воссоздаем ключ из пароля по умолчанию
                kdf = PBKDF2HMAC(
                    algorithm=hashes.SHA256(),
                    length=32,
                    salt=salt,
                    iterations=100000,
                )
                key = base64.urlsafe_b64encode(kdf.derive(master_password.encode()))
                
                # Загружаем ключ
                with open(self.key_file, "rb") as f:
                    stored_key = f.read()
                
                if key == stored_key:
                    self._fernet = Fernet(key)
                    print("✅ Ключ шифрования загружен (пароль по умолчанию)")
                    return
            except Exception:
                pass
            
            # Если ничего не подошло, создаем новый автоматически
            print("⚠️ Существующий ключ не подходит, создаю новый автоматически...")
            self._create_new_key_auto()
                
        except Exception as e:
            print(f"❌ Ошибка загрузки ключа шифрования: {e}")
            self._create_new_key_auto()
    
    def encrypt_api_key(self, api_key: str, provider: str = None) -> str:
        """Шифрует API ключ только для Web3 провайдеров"""
        # Web3 провайдеры, которые нужно шифровать
        web3_providers = ["Etherscan", "BSCScan", "CoinGecko", "0x"]
        
        # Если это не Web3 провайдер, возвращаем как есть (без шифрования)
        if provider and provider not in web3_providers:
            return api_key  # AI провайдеры не шифруются
        
        # Для Web3 провайдеров - шифруем
        if not self._fernet:
            return api_key  # Возвращаем как есть если шифрование не инициализировано
        
        try:
            encrypted = self._fernet.encrypt(api_key.encode())
            encrypted_str = base64.urlsafe_b64encode(encrypted).decode()
            return encrypted_str
        except Exception as e:
            print(f"❌ Ошибка шифрования API ключа {provider}: {e}")
            return api_key
    
    def decrypt_api_key(self, encrypted_api_key: str, provider: str = None) -> str:
        """Расшифровывает API ключ только для Web3 провайдеров"""
        # Web3 провайдеры, которые нужно расшифровывать
        web3_providers = ["Etherscan", "BSCScan", "CoinGecko", "0x"]
        
        # Если это не Web3 провайдер, возвращаем как есть (без расшифровки)
        if provider and provider not in web3_providers:
            return encrypted_api_key  # AI провайдеры не расшифровываются
        
        # Для Web3 провайдеров - расшифровываем
        if not self._fernet:
            return encrypted_api_key  # Возвращаем как есть если расшифровка не инициализирована
        
        try:
            encrypted_bytes = base64.urlsafe_b64decode(encrypted_api_key.encode())
            decrypted = self._fernet.decrypt(encrypted_bytes)
            decrypted_str = decrypted.decode()
            return decrypted_str
        except Exception as e:
            print(f"❌ Ошибка расшифровки API ключа {provider}: {e}")
            return encrypted_api_key
    
    def is_encrypted(self, api_key: str) -> bool:
        """Проверяет, зашифрован ли API ключ"""
        try:
            # Пытаемся расшифровать - если получается, значит зашифрован
            decrypted = self.decrypt_api_key(api_key)
            return decrypted != api_key
        except:
            return False

# Глобальный экземпляр шифрования
api_key_encryption = APIKeyEncryption()


